package OpenInteract::Handler::Package;

# $Id: Package.pm,v 1.7 2002/01/02 02:43:54 lachoy Exp $

use strict;
use Data::Dumper  qw( Dumper );
use OpenInteract::Handler::GenericDispatcher qw( DEFAULT_SECURITY_KEY );
use SPOPS::Secure qw( :level );

@OpenInteract::Handler::Package::ISA               = qw( OpenInteract::Handler::GenericDispatcher SPOPS::Secure );
$OpenInteract::Handler::Package::VERSION           = sprintf("%d.%02d", q$Revision: 1.7 $ =~ /(\d+)\.(\d+)/);
$OpenInteract::Handler::Package::author            = 'chris@cwinters.com';
$OpenInteract::Handler::Package::default_method    = 'listing';
@OpenInteract::Handler::Package::forbidden_methods = ();
%OpenInteract::Handler::Package::security          = (
 DEFAULT_SECURITY_KEY() => SEC_LEVEL_WRITE,
 listing                => SEC_LEVEL_READ );

use constant MAIN_SCRIPT => '/Package';

# List packages in the main interact directory as well as
# the packages in the website directory

sub listing {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $params = { main_script => MAIN_SCRIPT,
                   error_msg   => $p->{error_msg} };

    # First retrieve the packages from this app

    my $website_dir = $R->CONFIG->{dir}{base};
    $R->DEBUG && $R->scrib( 1, "Trying to open repository in website dir ($website_dir)" );
    my $website_repository  = $R->repository->fetch(
                                       undef, { directory => $website_dir } );
    $params->{website_pkg}  = $website_repository->fetch_all_packages();

    # Next, retrieve the packages from base installation

    my $base_install_dir = $R->CONFIG->{dir}{interact};
    $R->DEBUG && $R->scrib( 1, "Trying to open repository in base installation dir ($base_install_dir)" );
    my $base_repository     = $R->repository->fetch(
                                       undef, { directory => $base_install_dir } );
    $params->{interact_pkg} = $base_repository->fetch_all_packages();
    $R->{page}{title} = "Package Listing";
    return $R->template->handler( {}, $params,
                                  { name => 'base::package_listing' } );
}


# The only thing you can edit is the notes, so we don't need to go
# through contortions here

sub show {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $params = { main_script => MAIN_SCRIPT,
                   error_msg   => $p->{error_msg} };
    my $do_edit = ( $R->apache->param( 'edit' ) and $p->{level} >= SEC_LEVEL_WRITE );

    $params->{scope} = $R->apache->param( 'scope' ) || 'local';
    my $base_dir = ( $params->{scope} eq 'interact' )
                     ? $R->CONFIG->{dir}{interact}
                     : $R->CONFIG->{dir}{base};
    my $repository = $R->repository->fetch( undef, { directory => $base_dir } );

    my $name    = $R->apache->param( 'name' );
    my $version = $R->apache->param( 'version' );
    my $pkg = $p->{pkg} ||
              $repository->fetch_package_by_name({ name    => $name,
                                                   version => $version });
    unless ( $pkg ) {
        my $error_msg = "Sorry, couldn't find a package matching the criteria you specified.";
        return $class->listing({ error_msg => $error_msg });
    }
    if ( $version and $version != $pkg->{version} ) {
        $params->{error_msg} .= "Found a working package, but it wasn't the version you requested.";
    }
    $params->{pkg} = $pkg;
    return $R->template->handler( {}, $params,
                                  { name => 'base::package_detail' } );
}

sub edit {
    my ( $class, $p ) = @_;
    my $params = {};
    my $R = OpenInteract::Request->instance;
    my $name    = $R->apache->param( 'name' );
    my $version = $R->apache->param( 'version' );
    my $scope   = $R->apache->param( 'scope' );
    my $base_dir = ( $scope eq 'interact' ) ? $R->CONFIG->{dir}{interact} : $R->CONFIG->{dir}{base};
    my $repository = $R->repository->fetch( undef, { directory => $base_dir,
                                                     perm      => 'write' } );

    my $pkg = $repository->fetch_package_by_name({ name    => $name,
                                                   version => $version });
    if ( $@ or ! $pkg ) {
        my $error = $@ || "No package found using $name-$version";
        my $error_msg = "No changes made; cannot retrieve package for editing. Error: $error";
        return $class->listing({ error_msg => $error_msg });
    }
    my @editable_fields = qw( notes );
    foreach my $field ( @editable_fields ) {
        $pkg->{ $field } = $R->apache->param( $field );
    }
    $pkg->{last_updated_on} = $repository->now;
    $pkg->{last_updated_by} = join( ' ', $R->{auth}{user}{login_name}, "<$R->{auth}{user}{email}>" );
    $repository->save_package( $pkg );
    eval { $repository->save() };
    if ( $@ ) {
        $params->{error_msg} = "Cannot save package changes! Error returned: $@";
    }
    $params->{pkg} = $pkg;
    return $class->show( $params );
}


sub remove {
    my ( $class, $p ) = @_;
    my $msg = <<MSG;
Sorry, removal of packages is not supported yet. Please use the 
command line interface provided by <code>oi_manage</code>.
MSG
    return $class->listing({ error_msg => $msg });
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::Package - Allow viewing/editing of package information in base and website

=head1 DESCRIPTION

This handler simply displays information about packages in repository.

=head1 METHODS

B<listing>

Lists the packages in both the base installation package repository
and in the current website package repository.

B<show>

Displays details for a single package.

B<edit>

Edits information for a single package.

=head1 TO DO

B<Ensure editing works>

B<Be able to save more information per package>

=head1 BUGS

Need to test this more and shake any out.

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
