#!/usr/bin/perl -w
#
#    DateFormat.t
#
#    Copyright (c) 2000 Brian Stell
#
#    This package is free software and is provided ``as is'' without
#    express or implied warranty. It may be used, redistributed and/or
#    modified under the terms of the Perl Artistic License
#    (see http://www.perl.com/perl/misc/Artistic.html)
#

use strict;
use Test;
BEGIN { plan tests => 6 }
use ExtUtils::testlib;

use utf8;
use ICU qw(U_ZERO_ERROR);
use ICU::DecimalFormatSymbols qw(:ENumberFormatSymbol);
use ICU::Locale;

#
# Make sure we always use the same fallback locale
#
my $default_locale = new ICU::Locale('en_US');
my $euro = "\x{20AC}";
ICU::Locale->setDefault($default_locale);

my $tz_pst = ICU::TimeZone->createTimeZone('PST');
ICU::TimeZone->setDefault($tz_pst);

ok(check_enums(), 1);
ok(check_getDecimalSeparator(), 1);
ok(check_getSymbol(), 1);
ok(check_setSymbol(), 1);
ok(check_DecimalFormatSymbols__new(), 1);
ok(check_setDebugLevel(), 1);

exit(0);

########################################################################
#
# The subroutines and tests
#
########################################################################

sub check_enums {

    if (   (kDecimalSeparatorSymbol == 0)
        && (kGroupingSeparatorSymbol == 1)
        && (kPatternSeparatorSymbol == 2)
        && (kPercentSymbol == 3)
        && (kZeroDigitSymbol == 4)
        && (kDigitSymbol == 5)
        && (kMinusSignSymbol == 6)
        && (kPlusSignSymbol == 7)
        && (kCurrencySymbol == 8)
        && (kIntlCurrencySymbol == 9)
        && (kMonetarySeparatorSymbol == 10)
        && (kExponentialSymbol == 11)
        && (kPermillSymbol == 12)
        && (kPadEscapeSymbol == 13)
        && (kInfinitySymbol == 14)
        && (kNaNSymbol == 15)
        && (kFormatSymbolCount == 16)) {
        return(1);
    }
    else {
        return(0);
    }
}

sub check_getDecimalSeparator {
    my $result = 0;

    # check decimal separator for several common locales
    if  (  (check_getSymbol1('en_US', kDecimalSeparatorSymbol, '.'))
        && (check_getSymbol1('fr_FR', kDecimalSeparatorSymbol, ','))
        && (check_getSymbol1('de_DE', kDecimalSeparatorSymbol, ','))
        && (check_getSymbol1('nl_NL', kDecimalSeparatorSymbol, ','))
        ) {
        $result = 1;
    }
    return($result);
}

sub check_getSymbol {
    my $result = 0;

    # check decimal separator for several common locales
    if  (  (check_getSymbol1('en_US', kCurrencySymbol, '$'))
        && (check_getSymbol1('fr_FR', kCurrencySymbol, $euro))
        && (check_getSymbol1('de_DE', kCurrencySymbol, $euro))
        && (check_getSymbol1('nl_NL', kCurrencySymbol, $euro))
        ) {
        $result = 1;
    }
    return($result);
}

sub check_getSymbol1 {
    my ($locale_str, $symbol, $expected_result) = @_;
    my $result = 0;

    my $locale = new ICU::Locale($locale_str);
    ICU::Locale->setDefault($locale);
    my $df = new ICU::DecimalFormatSymbols();
    if (($df->getSymbol($symbol) eq $expected_result)) {
        $result = 1;
    }
    else {
        print "$locale_str getSymbol() = ("
                   .$df->getSymbol($symbol).") "
                   . "expect ($expected_result)\n";
    }
    ICU::Locale->setDefault($default_locale);
    return($result);
}

sub check_setSymbol {
    my $result = 0;

    # check decimal separator for several common locales
    if  (  (check_setSymbol1('en_US', kCurrencySymbol, 'xx'))
        && (check_setSymbol1('fr_FR', kCurrencySymbol, 'xx'))
        && (check_setSymbol1('de_DE', kCurrencySymbol, 'xx'))
        && (check_setSymbol1('nl_NL', kCurrencySymbol, 'xx'))
        ) {
        $result = 1;
    }
    return($result);
}

sub check_setSymbol1 {
    my ($locale_str, $symbol, $new_val) = @_;
    my $result = 0;

    my $locale = new ICU::Locale($locale_str);
    ICU::Locale->setDefault($locale);
    my $df = new ICU::DecimalFormatSymbols();
    $df->setSymbol($symbol, $new_val);
    if (($df->getSymbol($symbol) eq $new_val)) {
        $result = 1;
    }
    else {
        print "$locale_str failed to setSymbol() to ($new_val)\n";
    }
    ICU::Locale->setDefault($default_locale);
    return($result);
}

sub check_DecimalFormatSymbols__new {
    my $result = 1;

    my $status = U_ZERO_ERROR;
    my $locale = new ICU::Locale('en_US');
    my $dfs1 = new ICU::DecimalFormatSymbols();
    if (!$dfs1) {
        print "*** test 1 failed\n";
        $result = 0;
    }
    my $dfs2 = new ICU::DecimalFormatSymbols(\$status);
    if (!$dfs2) {
        print "*** test 2 failed\n";
        $result = 0;
    }
    my $dfs3 = new ICU::DecimalFormatSymbols($locale);
    if (!$dfs3) {
        print "*** test 3 failed\n";
        $result = 0;
    }
    $status = U_ZERO_ERROR;
    my $dfs4 = new ICU::DecimalFormatSymbols($locale, \$status);
    if (!$dfs4) {
        print "*** test 4 failed\n";
        $result = 0;
    }
    $status = int(1);
    my $dfs5 = new ICU::DecimalFormatSymbols($dfs1);
    if (!$dfs5) {
        print "*** test 5 failed\n";
        $result = 0;
    }
    ICU::DecimalFormatSymbols::setDebugLevel(0);

    return($result);
}

sub check_setDebugLevel {
    my $result = 1;
    my $old_debug_level1 = ICU::DecimalFormatSymbols::setDebugLevel();
    my $old_debug_level2 = ICU::DecimalFormatSymbols::setDebugLevel();
    if ($old_debug_level1 != $old_debug_level2) {
        $result = 0;
    }
    my $val1 = ICU::DecimalFormatSymbols::setDebugLevel(3);
    if ($val1 != $old_debug_level2) {
        $result = 0;
    }
    my $val2 = ICU::DecimalFormatSymbols::setDebugLevel();
    if ($val2 != 3) {
        $result = 0;
    }
    ICU::DecimalFormatSymbols::setDebugLevel($old_debug_level2);

    return($result);
}

