/*                                                                
**  Copyright (C) 1996,2007,2010,2019  Smithsonian Astrophysical Observatory 
*/                                                                

/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 3 of the License, or       */
/*  (at your option) any later version.                                     */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License along */
/*  with this program; if not, write to the Free Software Foundation, Inc., */
/*  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.             */
/*                                                                          */

/* rdbf.c
**
	Routines to manipulate rdb parameter files.
**/

#ifdef __linux__
#undef _XOPEN_SOURCE
#define _XOPEN_SOURCE 600
/* stdio.h: fileno, ftruncate, ftello */
#define _POSIX_C_SOURCE 2
#endif

#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "pfile.h"
#include "ptemplat.h"
#include "parameter.h"
#include "parament.h"
#include "putil.h"
#include "rdb.h"
#include "find.h"

static ParamList *RDBParRead (FILE * file, char *mode);
static void RDBParW (Parameter * p, FILE * f);

static void outputstr (FILE * f, char *str);


/*
	The mode argument:

		"l"	lock file for exclusive access.
*/

ParamList *
RDBParOpen (const char *psetname, const char *filename, const char *mode)
{
  FILE *f;
  ParamList *plist;
  char *fmode = strpbrk (mode, "w") == NULL ? "r" : "r+";

  if (filename == NULL)
    return NULL;

  if ((f =
       fopen (cxcFind (filename, fmode, ".rdb", PARAMPATH), fmode)) == NULL)
    return NULL;

  if (strpbrk (mode, "l"))
    Lock (f);

  if ((plist = RDBParFile (f)) == NULL)
    return (ParamList *) ParamFileCant;

  plist->file = f;
  plist->filename = CXCNewString (filename);
  plist->psetname = CXCNewString (psetname);
  plist->mode = CXCNewString (mode);
  plist->read = RDBParRead;
  plist->writ = RDBParWrit;
  plist->clos = NULL;

  return plist;
}


static ParamList *
RDBParRead (FILE * file, char *mode)
{
  if (strpbrk (mode, "l"))
    Lock (file);
  fseek (file, 0, 0);


  return RDBParFile (file);
}


void
RDBParWrit (FILE * file, ParamList * plist)
{
  int i;

  fseek (file, 0, 0);

  fprintf (file, "name\ttype\tmode\tvalue\tmin\tmax\tprompt\n");
  fprintf (file, "----\t----\t----\t-----\t---\t---\t------\n");

  for (i = 0; i < plist->nparam; i++)
    RDBParW (plist->param[i], file);

  fflush (file);
  ftruncate (fileno (file), ftello (file));
}

static void
RDBParW (Parameter * p, FILE * f)
{
  char namestr[SZ_PFLINE];
  char typestr[SZ_PFLINE];
  char modestr[SZ_PFLINE];
  char valustr[SZ_PFLINE];
  char minnstr[SZ_PFLINE];
  char maxxstr[SZ_PFLINE];
  char promstr[SZ_PFLINE];

  /* This fixes bug #14209 where the buffer passed in pvalue.value is larger
     than the receipient buffer which is SZ_PFLINE. at the VOnvert function
     the data is copied and caused a SegV - same as in iraf.c
  
     SL-98: Added check for FileType */
  if(p)
  {
    if((p->pvalue.value != NULL)&&((TypeIndex (p->pvalue.type) == StringType)
                                || (TypeIndex (p->pvalue.type) == FileType)))
    {
      if(strlen(p->pvalue.value) > SZ_PFLINE-1){
        char *r = (char *)( p->pvalue.value);
        r[SZ_PFLINE-1] = '\0';
      }
    }
  }

  ParamEntry (NULL, p, namestr, modestr, typestr, valustr, minnstr, maxxstr,
	      promstr);

  outputstr (f, namestr);
  putc ('\t', f);
  outputstr (f, typestr);
  putc ('\t', f);
  outputstr (f, modestr);
  putc ('\t', f);
  outputstr (f, valustr);
  putc ('\t', f);
  outputstr (f, minnstr);
  putc ('\t', f);
  outputstr (f, maxxstr);
  putc ('\t', f);
  outputstr (f, promstr);
  putc ('\n', f);
}

void
outputstr (FILE * f, char *str)
{
  while (*str)
    {
      switch (*str)
	{
	case '\t':
	  putc ('\\', f);
	  putc ('t', f);
	  break;
	case '\n':
	  putc ('\\', f);
	  putc ('n', f);
	  break;
	default:
	  putc (*str, f);
	  break;

	}
      str++;
    }
}
