package Database::Temp::Driver::CSV;

use strict;
use warnings;

# ABSTRACT: Create an ad-hoc database, driver for CSV

our $VERSION = '0.003'; # VERSION: generated by DZP::OurPkgVersion

use Module::Load::Conditional qw( can_load );

use Carp       qw( shortmess );
use File::Path qw( make_path remove_tree );
use File::Spec;

use Log::Any;
use Try::Tiny;
use Const::Fast;

const my $TEMP_PATH_CHMOD => 700;

sub is_available {
    my $_log   = Log::Any->get_logger( category => 'Database::Temp' );
    my %needed = ( 'DBD::CSV' => 0.57, );
    if ( !can_load( modules => \%needed ) ) {
        $_log->infof( 'Cannot load module %s, %s', %needed );
        return 0;
    }
    return 1;
}

sub new {
    my ( $class, %params ) = @_;
    my $_log = Log::Any->get_logger( category => 'Database::Temp' );

    my $dir     = $params{'args'}->{'dir'} // File::Spec->tmpdir();
    my $db_dir  = $params{'name'};
    my $dirpath = File::Spec->catfile( $dir, $db_dir );
    make_path( $dirpath, { verbose => 0, mode => oct($TEMP_PATH_CHMOD), } );
    my $dsn = 'dbi:CSV:';
    $_log->debugf( 'Created temp dirpath \'%s\'', $dirpath );

    my %attrs = (
        'AutoCommit' => 1,
        'RaiseError' => 1,
        'PrintError' => 1,
        'RaiseWarn'  => 1,
        'PrintWarn'  => 1,
        'TaintIn'    => 1,
        'TaintOut'   => 0,
        'TraceLevel' => 0,
        'f_dir'      => $dirpath,
        'f_ext'      => '.csv/r',
        'f_encoding' => 'utf8',
    );
    my %info = ( 'dirpath' => $dirpath, );

    # Construct start method
    my $_start = sub {
        my ( $dbh, $name ) = @_;
        Log::Any->get_logger( category => 'Database::Temp' )->debugf( 'Created temp db \'%s\'', $name );
    };

    # Construct init method
    my $init;
    if ( ref $params{'init'} eq 'CODE' ) {
        $init = $params{'init'};
    }
    else {    # SCALAR
              # Attn. CSV does not have transactions (begin_work, commit)
        $init = sub {
            my ($dbh) = @_;
            foreach my $row ( split qr/;\s*/msx, $params{'init'} ) {
                $dbh->do($row);
            }
            return;
        }
    }

    # Construct deinit method
    my $deinit;
    if ( ref $params{'deinit'} eq 'CODE' ) {
        $deinit = $params{'deinit'};
    }
    else {    # SCALAR
        $deinit = sub {
            my ($dbh) = @_;
            foreach my $row ( split qr/;\s*/msx, $params{'deinit'} ) {
                $dbh->do($row);
            }
            return;
        }
    }

    # Construct _cleanup method
    my $_cleanup = sub {
        my ( $dbh, $name, $info ) = @_;
        $_log->infof( 'Deleting dir %s for temp db \'%s\'', $info->{'dirpath'}, $name );
        remove_tree( $info->{'dirpath'}, { safe => 1, } );
    };

    # Create database representing object.
    return Database::Temp::DB->new(
        driver   => ( __PACKAGE__ =~ m/^Database::Temp::Driver::(.*)$/msx )[0],
        name     => $params{'name'},
        cleanup  => $params{'cleanup'} // 0,
        _cleanup => $_cleanup,
        _start   => $_start,
        init     => $init,
        deinit   => $deinit,
        dsn      => $dsn,
        username => undef,
        password => undef,
        attr     => \%attrs,
        info     => \%info,
    );
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Database::Temp::Driver::CSV - Create an ad-hoc database, driver for CSV

=head1 VERSION

version 0.003

=head2 is_available

Can this driver provide a database?

Return boolean.

=head2 new

Create a temp database.

User should never call this subroutine directly, only via L<Database::Temp>.

=head1 AUTHOR

Mikko Koivunalho <mikkoi@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2023 by Mikko Johannes Koivunalho.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
