package main

import (
	"fmt"
	"log"
	"os"
	"regexp"
	"strings"
)

const MISSING_PKGCONF_DEPENDENCY_TAG = "missing-pkgconfig-dependency"

type pkgconfErr struct {
	debPkg     string
	pkg        string
	missingDep string
}

func (p pkgconfErr) Error() string {
	return fmt.Sprintf("%s: %s %s => %s", p.debPkg, MISSING_PKGCONF_DEPENDENCY_TAG, p.pkg, p.missingDep)
}

type pkgconfDepChecker struct {
	emit bool
}

func newPkgconfDepChecker(tags tagFilter) pkgconfDepChecker {
	return pkgconfDepChecker{
		emit: tags.shouldEmit(MISSING_PKGCONF_DEPENDENCY_TAG),
	}
}

func (pd pkgconfDepChecker) check(pkg2files map[string][]string) []error {
	if !pd.emit || !pathExists("/usr/bin/pkgconf") {
		return nil
	}

	ignorePath := regexp.MustCompile("^(/usr/(?:share|lib(?:/[^/]+)?)/pkgconfig/)([^/]+)[.]pc$")
	// pkgs maps from a pc name to a binary package name. The pc name
	// maps to a source package but does not necessarily match the source
	// package name.
	pkgs := make(map[string]string)
	pkgCfgPaths := newStringSet()
	for debPkg, files := range pkg2files {
		for _, f := range files {
			m := ignorePath.FindStringSubmatch(f)
			if len(m) < 3 {
				continue
			}
			pkgCfgPaths.add(m[1])
			pkg := m[2]
			pkgs[pkg] = debPkg
		}
	}

	if err := os.Setenv("PKG_CONFIG_PATH", pkgCfgPaths.String()); err != nil {
		verboseLog(fmt.Sprintf(
			"Skipping %s checks: unable to set environment variable PKG_CONFIG_PATH",
			MISSING_PKGCONF_DEPENDENCY_TAG))
		return nil
	}

	var tags []error
	for pkg, debPkg := range pkgs {
		out, err := runCommand(append(
			strings.Split("/usr/bin/pkgconf --exists --print-errors", " "), pkg))
		if err == nil {
			continue
		}

		missingPkg := regexp.MustCompile("Package '(.+)', required by '" + pkg + "', not found")
		m := missingPkg.FindStringSubmatch(strings.Join(out, " "))
		if len(m) < 2 {
			log.Fatalf("Unexpected pkgconf output: %v", out)
		}
		depPkg := m[1]
		tags = append(tags, pkgconfErr{
			debPkg:     debPkg,
			pkg:        pkg,
			missingDep: string(depPkg),
		})
	}

	return tags
}
