"""
Copyright (c) 2003-2007, Paul Varga and other contributors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation
      and/or other materials provided with the distribution.
    * Neither the name of the project nor the names of its contributors may be
      used to endorse or promote products derived from this software without
      specific prior written permission.
"""

import cairo, gtk, gtk.gdk, struct

def render_progress_icon(icon_path, width, height, fraction, style=None):
	def gdk_to_color4f(color, max=65535.0, alpha=1.0):
		return (color.red / max, color.green / max, color.blue / max, alpha)
	def cairo_imagesurface_to_pixbuf(s):
		def clamp(value, min, max):
			if value < min:
				return min
			elif value > max:
				return max
			else:
				return value
		# I don't think this is necessary, but just in-case...
		clamp_byte = lambda value: clamp(value, 0x00, 0xFF)
		
		# Assumptions...
		assert s.get_format() == cairo.FORMAT_ARGB32
		size = 4
		
		sdata, width, height, stride = s.get_data(), s.get_width(), s.get_height(), s.get_stride()
		data = []
		
		for y in range(0, height):
			row = y*stride
			
			for x in range(0, width):
				at = size*x + row
				
				argb = struct.unpack('I', sdata[at:at+4])[0]
				a = (argb & 0xFF000000) >> 24
				r = (argb & 0x00FF0000) >> 16
				g = (argb & 0x0000FF00) >> 8
				b = (argb & 0x000000FF)
				if a > 0x00:
					data.append(struct.pack('BBBB',
						clamp_byte(int(0xFF/float(a) * r)),
						clamp_byte(int(0xFF/float(a) * g)),
						clamp_byte(int(0xFF/float(a) * b)),
						a))
				else:
					data.append('\x00\x00\x00\x00')
		
		return gtk.gdk.pixbuf_new_from_data(''.join(data), gtk.gdk.COLORSPACE_RGB, True, 8, width, height, size*width)
	def draw_shadow_in(c, style, state, x, y, width, height):
		#
		# dddddddl	d = dark
		# d00000bl	l = light
		# d0    bl	0 = black
		# dbbbbbbl	b = bg
		# llllllll
		#
		c.save()
		
		# l
		c.set_source_rgba(*gdk_to_color4f(style.light[state]))
		
		c.set_line_width(style.ythickness)
		c.move_to(x, y+height-1.0)
		c.line_to(x+width-1.0, y+height-1.0)
		c.stroke()
		
		c.set_line_width(style.xthickness)
		c.move_to(x+width-1.0, y)
		c.line_to(x+width-1.0, y+height-1.0)
		c.stroke()
		
		# b
		c.set_source_rgba(*gdk_to_color4f(style.bg[state]))
		
		c.set_line_width(style.ythickness)
		c.move_to(x+1.0, y+height-2.0)
		c.line_to(x+width-2.0, y+height-2.0)
		c.stroke()
		
		c.set_line_width(style.xthickness)
		c.move_to(x+width-2.0, y+1.0)
		c.line_to(x+width-2.0, y+height-2.0)
		c.stroke()
		
		# 0
		c.set_source_rgba(*gdk_to_color4f(style.black))
		
		c.set_line_width(style.ythickness)
		c.move_to(x+1.0, y+1.0)
		c.line_to(x+width-2.0, y+1.0)
		c.stroke()
		
		c.set_line_width(style.xthickness)
		c.move_to(x+1.0, y+1.0)
		c.line_to(x+1.0, y+height-2.0)
		c.stroke()
		
		# d
		c.set_source_rgba(*gdk_to_color4f(style.dark[state]))
		
		c.set_line_width(style.ythickness)
		c.move_to(x, y)
		c.line_to(x+width-1.0, y)
		c.stroke()
		
		c.set_line_width(style.xthickness)
		c.move_to(x, y)
		c.line_to(x, y+height-1.0)
		c.stroke()
		
		c.restore()
	def draw_progress(c, style, x, y, width, height, fraction):
		# progress bar
		c.save()
		c.set_source_rgba(*gdk_to_color4f(style.bg[gtk.STATE_NORMAL]))
		c.rectangle(x, y, width, height)
		c.fill()
		
		c.set_source_rgba(*gdk_to_color4f(style.bg[gtk.STATE_SELECTED]))
		c.rectangle(x, y, width*fraction, height)
		c.fill()
		c.restore()
	
	if style is None:
		# FIXME This is a horrible hack. Is there a better way?
		window = gtk.Window()
		progressbar = gtk.ProgressBar()
		window.add(progressbar)
		progressbar.realize()
		style = progressbar.get_style()
	
	progress_height = max(int(height/ 3.0), min(5, height)) # height/3.0 bounded by [5, size]
	progress_y      = (height - progress_height) * 0.5
	
	s = cairo.ImageSurface(cairo.FORMAT_ARGB32, width, height)
	c = cairo.Context(s)
	
	# progress
	draw_progress(c, style, 0, progress_y, width, progress_height, fraction)
	
	# shadow
	draw_shadow_in(c, style, gtk.STATE_NORMAL, 0, progress_y, width, progress_height)
	
	final_pixbuf    = gtk.gdk.pixbuf_new_from_file_at_size(icon_path, width, height)
	progress_pixbuf = cairo_imagesurface_to_pixbuf(s)
	progress_pixbuf.composite(final_pixbuf, 0, 0, width, height, 0, 0, 1.0, 1.0, gtk.gdk.INTERP_BILINEAR, 0xCC)
	
	return final_pixbuf
