/*
     This file is part of GNUnet
     (C) 2005, 2006, 2010, 2011 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/lib/about.c
 * @brief code to display an about dialog
 * @author Christian Grothoff
 * @author Igor Wronsky
 */
#include "gnunet_gtk.h"

struct AboutDialogContext
{
  GtkBuilder *builder;
  GtkWidget *ad;
  GtkWidget *about_credits_notebook;
  GtkWidget *about_license_scroller;
  GtkTextBuffer *license_contents;
};

static void
destroy_about_dialog (struct AboutDialogContext *ctx)
{
  gtk_widget_destroy (ctx->ad);
  g_object_unref (G_OBJECT (ctx->builder));
  GNUNET_free (ctx);
}


static void
about_window_show_exclusively (struct AboutDialogContext *ctx, gchar * name)
{
  if (name == NULL)
  {
    gtk_widget_hide (ctx->about_credits_notebook);
    gtk_widget_hide (ctx->about_license_scroller);
  }
  else if (strcmp ("about_credits_notebook", name) == 0)
  {
    gtk_widget_show (ctx->about_credits_notebook);
    gtk_widget_hide (ctx->about_license_scroller);
  }
  else if (strcmp ("about_license_scroller", name) == 0)
  {
    gtk_widget_show (ctx->about_license_scroller);
    gtk_widget_hide (ctx->about_credits_notebook);
  }
}


G_MODULE_EXPORT void
GNUNET_GTK_about_window_realized (GtkWidget * widget, struct AboutDialogContext *ctx)
{
  gchar *license = NULL;
  const char *path;
  char *license_path;

  path = GNUNET_OS_installation_get_path (GNUNET_OS_IPK_DOCDIR);
  if (path != NULL)
    GNUNET_asprintf (&license_path, "%s%s", path, "COPYING");
  else
    license_path = GNUNET_strdup ("COPYING");
  if (g_file_get_contents (license_path, &license, NULL, NULL) &&
      (license != NULL))
  {
    gtk_text_buffer_set_text (ctx->license_contents, license, -1);
    g_free (license);
  }
  GNUNET_free_non_null (license_path);
}


G_MODULE_EXPORT void
GNUNET_GTK_about_close_button_clicked (GtkButton * widget, struct AboutDialogContext *ctx)
{
  destroy_about_dialog (ctx);
}


G_MODULE_EXPORT gboolean
GNUNET_GTK_about_window_got_delete_event (GtkWidget * widget, GdkEvent * event,
                                          struct AboutDialogContext *ctx)
{
  destroy_about_dialog (ctx);
  return FALSE;
}


G_MODULE_EXPORT void
GNUNET_GTK_about_credits_button_clicked (GtkButton * widget, struct AboutDialogContext *ctx)
{
  about_window_show_exclusively (ctx, "about_credits_notebook");
}

G_MODULE_EXPORT void
GNUNET_GTK_about_license_button_clicked (GtkButton * widget, struct AboutDialogContext *ctx)
{
  about_window_show_exclusively (ctx, "about_license_scroller");
}


/**
 * This displays an about dialog.
 *
 * @param dialogfile name of the glade file containing the dialog
 * @param dialogname name of the about dialog in the file
 */
void
GNUNET_GTK_display_about (const char *dialogfile)
{
  struct AboutDialogContext *ctx;
  ctx = GNUNET_malloc (sizeof (struct AboutDialogContext));
  ctx->builder = GNUNET_GTK_get_new_builder (dialogfile, ctx);

  if (ctx->builder == NULL)
  {
    GNUNET_free (ctx);
    return;
  }
  ctx->about_credits_notebook =
      GTK_WIDGET (gtk_builder_get_object (ctx->builder, "about_credits_notebook"));
  ctx->about_license_scroller =
      GTK_WIDGET (gtk_builder_get_object (ctx->builder, "about_license_scroller"));
  ctx->ad = GTK_WIDGET (gtk_builder_get_object (ctx->builder, "about_window"));
  ctx->license_contents =
      GTK_TEXT_BUFFER (gtk_builder_get_object (ctx->builder, "license_contents"));
  gtk_widget_show (ctx->ad);
}


/* end of about.c */
