/*
 * Copyright 2021-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.schema;

import org.apiguardian.api.API;

import java.util.Objects;

import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * A qualified name consisting of a namespace and a simple name.
 *
 * @since 0.1.0
 */
@API(status = EXPERIMENTAL, since = "0.1.0")
public class QualifiedName {

	/**
	 * Create the qualified name for the supplied namespace and name.
	 *
	 * @param namespace the namespace
	 * @param simpleName the simple name
	 * @return namespace with the supplied URI
	 */
	public static QualifiedName of(Namespace namespace, String simpleName) {
		return new QualifiedName(namespace, simpleName);
	}

	private final Namespace namespace;
	private final String simpleName;

	private QualifiedName(Namespace namespace, String simpleName) {
		this.namespace = namespace;
		this.simpleName = simpleName;
	}

	/**
	 * {@return the namespace of this qualified name}
	 */
	public Namespace getNamespace() {
		return namespace;
	}

	/**
	 * {@return the simple name of this qualified name}
	 */
	public String getSimpleName() {
		return simpleName;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (o == null || getClass() != o.getClass()) {
			return false;
		}
		QualifiedName that = (QualifiedName) o;
		return namespace.equals(that.namespace) && simpleName.equals(that.simpleName);
	}

	@Override
	public int hashCode() {
		return Objects.hash(namespace, simpleName);
	}
}
