// license:BSD-3-Clause
// copyright-holders:David Haywood

#include "emu.h"
#include "arcompact_helper.ipp"

inline uint32_t arcompact_device::branch_common(uint16_t op, bool cond, unsigned width)
{
	if (cond)
	{
		int s = util::sext(op, width);
		uint32_t realaddress = (m_pc & 0xfffffffc) + (s * 2);
		return realaddress;
	}
	return m_pc + 2;
}

// #######################################################################################################################
//                                 IIII I    s
// BREQ_S b,0,s8                   1110 1bbb 0sss ssss
// #######################################################################################################################

uint32_t arcompact_device::handleop_BREQ_S_b_0_s8(uint16_t op) // BREQ_S b,0,s8
{
	uint8_t breg = common16_get_and_expand_breg(op);
	return branch_common(op, !m_regs[breg], 7);
}

// #######################################################################################################################
//                                 IIII I    s
// BRNE_S b,0,s8                   1110 1bbb 1sss ssss
// #######################################################################################################################

uint32_t arcompact_device::handleop_BRNE_S_b_0_s8(uint16_t op) // BRNE_S b,0,s8
{
	uint8_t breg = common16_get_and_expand_breg(op);
	return branch_common(op, m_regs[breg], 7);
}

// #######################################################################################################################
//                                 IIII ISS
// B_S s10                         1111 000s ssss ssss
// #######################################################################################################################

uint32_t arcompact_device::handleop_B_S_s10(uint16_t op) // B_S s10  (branch always)
{
	return branch_common(op, true, 9);
}

// #######################################################################################################################
//                                 IIII ISS
// BEQ_S s10                       1111 001s ssss ssss
// #######################################################################################################################

uint32_t arcompact_device::handleop_BEQ_S_s10(uint16_t op) // BEQ_S s10 (branch is zero bit is set)
{
	return branch_common(op ,status32_check_z(), 9);
}

// #######################################################################################################################
//                                 IIII ISS
// BNE_S s10                       1111 010s ssss ssss
// #######################################################################################################################

uint32_t arcompact_device::handleop_BNE_S_s10(uint16_t op) // BNE_S s10  (branch if zero bit isn't set)
{
	return branch_common(op ,!status32_check_z(), 9);
}

// #######################################################################################################################
//                                 IIII ISSs ss
// BGT_S s7                        1111 0110 00ss ssss
// #######################################################################################################################

uint32_t arcompact_device::handleop_BGT_S_s7(uint16_t op)
{
	return branch_common(op, condition_GT(), 6);
}

// #######################################################################################################################
//                                 IIII ISSs ss
// BGE_S s7                        1111 0110 01ss ssss
// #######################################################################################################################

uint32_t arcompact_device::handleop_BGE_S_s7(uint16_t op)
{
	return branch_common(op, condition_GE(), 6);
}

// #######################################################################################################################
//                                 IIII ISSs ss
// BLT_S s7                        1111 0110 10ss ssss
// #######################################################################################################################

uint32_t arcompact_device::handleop_BLT_S_s7(uint16_t op) // BLT_S
{
	return branch_common(op, condition_LT(), 6);
}

// #######################################################################################################################
//                                 IIII ISSs ss
// BLE_S s7                        1111 0110 11ss ssss
// #######################################################################################################################

uint32_t arcompact_device::handleop_BLE_S_s7(uint16_t op) // BLE_S
{
	return branch_common(op, condition_LE(), 6);
}

// #######################################################################################################################
//                                 IIII ISSs ss
// BHI_S s7                        1111 0111 00ss ssss
// #######################################################################################################################

uint32_t arcompact_device::handleop_BHI_S_s7(uint16_t op)
{
	return branch_common(op, condition_HI(), 6);
}

// #######################################################################################################################
//                                 IIII ISSs ss
// BHS_S s7                        1111 0111 01ss ssss
// #######################################################################################################################

uint32_t arcompact_device::handleop_BHS_S_s7(uint16_t op)
{
	return branch_common(op, condition_HS(), 6);
}

// #######################################################################################################################
//                                 IIII ISSs ss
// BLO_S s7                        1111 0111 10ss ssss
// #######################################################################################################################

uint32_t arcompact_device::handleop_BLO_S_s7(uint16_t op)
{
	return branch_common(op, condition_CS(), 6);
}

// #######################################################################################################################
//                                 IIII ISSs ss
// BLS_S s7                        1111 0111 11ss ssss
// #######################################################################################################################

uint32_t arcompact_device::handleop_BLS_S_s7(uint16_t op)
{
	return branch_common(op, condition_LS(), 6);
}

// #######################################################################################################################
//                                 IIII I
// BL_S s13                        1111 1sss ssss ssss
// #######################################################################################################################

uint32_t arcompact_device::handleop_BL_S_s13(uint16_t op) // BL_S s13
{
	int s = util::sext(op, 11);
	uint32_t realaddress = (m_pc & 0xfffffffc) + (s * 4);
	m_regs[REG_BLINK] = m_pc + 2;
	return realaddress;
}
