//===-- SwiftLanguageRuntimeRemoteAST.cpp --------------------------------===//
//
// This source file is part of the Swift.org open source project
//
// Copyright (c) 2014 - 2020 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See https://swift.org/LICENSE.txt for license information
// See https://swift.org/CONTRIBUTORS.txt for the list of Swift project authors
//
//===----------------------------------------------------------------------===//

#include "Plugins/TypeSystem/Swift/SwiftASTContext.h"
#include "LLDBMemoryReader.h"
#include "ReflectionContextInterface.h"
#include "SwiftLanguageRuntime.h"
#include "lldb/Symbol/VariableList.h"
#include "lldb/Utility/LLDBLog.h"
#include "lldb/Utility/Timer.h"
#include "swift/AST/ASTContext.h"
#include "swift/AST/ASTMangler.h"
#include "swift/AST/ASTWalker.h"
#include "swift/AST/Type.h"
#include "swift/AST/Types.h"
#include "swift/RemoteAST/RemoteAST.h"

using namespace lldb;
using namespace lldb_private;

namespace {
class ASTVerifier : public swift::ASTWalker {
  bool hasMissingPatterns = false;

  PreWalkAction walkToDeclPre(swift::Decl *D) override {
    if (auto *PBD = llvm::dyn_cast<swift::PatternBindingDecl>(D)) {
      if (PBD->getPatternList().empty()) {
        hasMissingPatterns = true;
        return Action::SkipChildren();
      }
    }
    return Action::Continue();
  }

public:
  /// Detect (one form of) incomplete types. These may appear if
  /// member variables have Clang-imported types that couldn't be
  /// resolved.
  static bool Verify(swift::Decl *D) {
    if (!D)
      return false;

    ASTVerifier verifier;
    D->walk(verifier);
    return !verifier.hasMissingPatterns;
  }
};

} // namespace

namespace lldb_private {

swift::remoteAST::RemoteASTContext &
SwiftLanguageRuntime::GetRemoteASTContext(SwiftASTContext &swift_ast_ctx) {
  // If we already have a remote AST context for this AST context,
  // return it.
  ThreadSafeASTContext ast_ctx = swift_ast_ctx.GetASTContext();
  auto known = m_remote_ast_contexts.find(*ast_ctx);
  if (known != m_remote_ast_contexts.end())
    return *known->second;

  // Initialize a new remote AST context.
  (void)GetReflectionContext();
  auto remote_ast_up = std::make_unique<swift::remoteAST::RemoteASTContext>(
      **ast_ctx, GetMemoryReader());
  auto &remote_ast = *remote_ast_up;
  m_remote_ast_contexts.insert(
      {*ast_ctx, std::move(remote_ast_up)});
  return remote_ast;
}

void SwiftLanguageRuntime::ReleaseAssociatedRemoteASTContext(
    swift::ASTContext *ctx) {
  m_remote_ast_contexts.erase(ctx);
}

std::optional<uint64_t> SwiftLanguageRuntime::GetMemberVariableOffsetRemoteAST(
    CompilerType instance_type, ValueObject *instance,
    llvm::StringRef member_name) {
  auto scratch_ctx =
      instance_type.GetTypeSystem().dyn_cast_or_null<SwiftASTContext>();
  if (!scratch_ctx || scratch_ctx->HasFatalErrors())
    return {};

  auto *remote_ast = &GetRemoteASTContext(*scratch_ctx);
  // Check whether we've already cached this offset.
  swift::TypeBase *swift_type =
      scratch_ctx->GetCanonicalSwiftType(instance_type).getPointer();
  if (!swift_type)
    return {};

  // Perform the cache lookup.
  MemberID key{swift_type, ConstString(member_name).GetCString()};
  auto it = m_member_offsets.find(key);
  if (it != m_member_offsets.end())
    return it->second;

  // Dig out metadata describing the type, if it's easy to find.
  // FIXME: the Remote AST library should make this easier.
  swift::remote::RemoteAddress optmeta;
  const swift::TypeKind type_kind = swift_type->getKind();
  switch (type_kind) {
  case swift::TypeKind::Class:
  case swift::TypeKind::BoundGenericClass: {
    LLDB_LOGF(GetLog(LLDBLog::Types),
              "[MemberVariableOffsetResolver] type is a class - trying to "
              "get metadata for valueobject %s",
              (instance ? instance->GetName().AsCString() : "<null>"));
    if (instance) {
      lldb::addr_t pointer = instance->GetPointerValue();
      if (!pointer || pointer == LLDB_INVALID_ADDRESS)
        break;
      auto address = swift::remote::RemoteAddress(
          pointer, swift::remote::RemoteAddress::DefaultAddressSpace);
      if (auto metadata = remote_ast->getHeapMetadataForObject(address))
        optmeta = metadata.getValue();
    }
    LLDB_LOGF(GetLog(LLDBLog::Types),
              "[MemberVariableOffsetResolver] optmeta = 0x%" PRIx64,
              optmeta.getRawAddress());
    break;
  }

  default:
    // Bind generic parameters if necessary.
    if (instance && swift_type->hasTypeParameter())
      if (auto *frame = instance->GetExecutionContextRef().GetFrameSP().get())
        if (auto bound = llvm::expectedToOptional(
                             BindGenericTypeParameters(*frame, instance_type))
                             .value_or(CompilerType())) {
          LLDB_LOGF(
              GetLog(LLDBLog::Types),
              "[MemberVariableOffsetResolver] resolved non-class type = %s",
              bound.GetTypeName().AsCString());

          swift_type = scratch_ctx->GetCanonicalSwiftType(bound).getPointer();
          MemberID key{swift_type, ConstString(member_name).GetCString()};
          auto it = m_member_offsets.find(key);
          if (it != m_member_offsets.end())
            return it->second;
        }
  }

  if (!swift_type)
    return {};

  // Try to determine whether it is safe to use RemoteAST.  RemoteAST
  // is faster than RemoteMirrors, but can't do dynamic types (checked
  // inside RemoteAST) or incomplete types (checked here).
  bool safe_to_use_remote_ast = true;
  if (swift::Decl *type_decl = swift_type->getNominalOrBoundGenericNominal())
    safe_to_use_remote_ast &= ASTVerifier::Verify(type_decl);

  // Use RemoteAST to determine the member offset.
  if (safe_to_use_remote_ast) {
    swift::remoteAST::Result<uint64_t> result =
        remote_ast->getOffsetOfMember(swift_type, optmeta, member_name);
    if (result) {
      LLDB_LOGF(GetLog(LLDBLog::Types),
                "[MemberVariableOffsetResolver] offset discovered = %" PRIu64,
                (uint64_t)result.getValue());

      // Cache this result.
      MemberID key{swift_type, ConstString(member_name).GetCString()};
      m_member_offsets.insert({key, result.getValue()});
      return result.getValue();
    }

    const auto &failure = result.getFailure();
    LLDB_LOGF(GetLog(LLDBLog::Types),
              "[MemberVariableOffsetResolver] failure: %s",
              failure.render().c_str());
  }
  return {};
}

#ifndef NDEBUG
ConstString SwiftLanguageRuntime::GetDynamicTypeName_ClassRemoteAST(
    ValueObject &in_value, lldb::addr_t instance_ptr) {
  // Dynamic type resolution in RemoteAST might pull in other Swift modules, so
  // use the scratch context where such operations are legal and safe.

  auto stack_frame_sp = in_value.GetExecutionContextRef().GetFrameSP();
  if (!stack_frame_sp)
    return {};
  auto &sc = stack_frame_sp->GetSymbolContext(eSymbolContextFunction);
  auto ts = TypeSystemSwiftTypeRefForExpressions::GetForTarget(
      in_value.GetTargetSP());
  if (!ts)
    return {};
  SwiftASTContextSP swift_ast_ctx = ts->GetSwiftASTContext(sc);
  if (!swift_ast_ctx)
    return {};

  auto &remote_ast = GetRemoteASTContext(*swift_ast_ctx);
  auto remote_ast_metadata_address =
      remote_ast.getHeapMetadataForObject(swift::remote::RemoteAddress(
          instance_ptr, swift::remote::RemoteAddress::DefaultAddressSpace));
  if (remote_ast_metadata_address) {
    auto instance_type = remote_ast.getTypeForRemoteTypeMetadata(
        remote_ast_metadata_address.getValue(),
        /*skipArtificial=*/true);
    if (instance_type) {
      auto ref_type = ToCompilerType(instance_type.getValue());
      ConstString name = ref_type.GetMangledTypeName();
      if (!name)
        LLDB_LOG(GetLog(LLDBLog::Types), "could not get type metadata:{0}\n",
                 instance_type.getFailure().render());
      return name;
    }
  }
  return {};
}

std::optional<std::pair<CompilerType, Address>>
SwiftLanguageRuntime::GetDynamicTypeAndAddress_ExistentialRemoteAST(
    ValueObject &in_value, CompilerType existential_type, bool use_local_buffer,
    lldb::addr_t existential_address) {
  // Dynamic type resolution in RemoteAST might pull in other Swift
  // modules, so use the scratch context where such operations are
  // legal and safe.
  auto scratch_ctx = TypeSystemSwiftTypeRefForExpressions::GetForTarget(
      in_value.GetTargetSP());
  if (!scratch_ctx)
    return {};

  auto stack_frame_sp = in_value.GetExecutionContextRef().GetFrameSP();
  if (!stack_frame_sp)
    return {};
  auto &sc = stack_frame_sp->GetSymbolContext(eSymbolContextFunction);
  SwiftASTContextSP swift_ast_ctx = scratch_ctx->GetSwiftASTContext(sc);
  if (!swift_ast_ctx)
    return {};

  auto remote_existential = swift::remote::RemoteAddress(
      existential_address, swift::remote::RemoteAddress::DefaultAddressSpace);
  auto &remote_ast = GetRemoteASTContext(*swift_ast_ctx);
  auto swift_type =
      llvm::expectedToStdOptional(swift_ast_ctx->GetSwiftType(existential_type))
          .value_or(swift::Type());
  if (!swift_type)
    return {};
  MemoryReaderLocalBufferHolder holder;
  if (use_local_buffer)
    holder = PushLocalBuffer(
        existential_address,
        llvm::expectedToOptional(in_value.GetByteSize()).value_or(0));

  auto result = remote_ast.getDynamicTypeAndAddressForExistential(
      remote_existential, swift_type);

  if (!result.isSuccess())
    return {};

  auto type_and_address = result.getValue();

  CompilerType type = ToCompilerType(type_and_address.InstanceType);
  Address address;
  address.SetRawAddress(type_and_address.PayloadAddress.getRawAddress());
  return {{type, address}};
}
#endif

llvm::Expected<CompilerType>
SwiftLanguageRuntime::BindGenericTypeParametersRemoteAST(
    StackFrame &stack_frame, CompilerType base_type) {
  // If this is a TypeRef type, bind that.
  auto sc = stack_frame.GetSymbolContext(lldb::eSymbolContextEverything);
  if (auto ts =
          base_type.GetTypeSystem().dyn_cast_or_null<TypeSystemSwiftTypeRef>())
    return BindGenericTypeParameters(stack_frame, *ts,
                                     base_type.GetMangledTypeName());

  Status error;
  auto &target = GetProcess().GetTarget();

  // A failing Clang import in a module context permanently damages
  // that module context.  Binding archetypes can trigger an import of
  // another module, so switch to a scratch context where such an
  // operation is safe.
  auto scratch_ctx = TypeSystemSwiftTypeRefForExpressions::GetForTarget(target);
  if (!scratch_ctx)
    return base_type;

  SwiftASTContextSP swift_ast_ctx = scratch_ctx->GetSwiftASTContext(sc);
  if (!swift_ast_ctx)
    return base_type;
  base_type = swift_ast_ctx->ImportType(base_type, error);

  if (base_type.GetTypeInfo() & lldb::eTypeIsSwift) {
    swift::Type target_swift_type(
        llvm::expectedToStdOptional(swift_ast_ctx->GetSwiftType(base_type))
            .value_or(swift::Type()));
    if (target_swift_type->hasArchetype())
      target_swift_type = target_swift_type->mapTypeOutOfContext().getPointer();

    ThreadSafeASTContext ast_ctx = swift_ast_ctx->GetASTContext();
    if (!ast_ctx)
      return base_type;

    // Replace opaque types with their underlying types when possible.
    swift::Mangle::ASTMangler mangler(**ast_ctx, true);
    // Rewrite all dynamic self types to their static self types.
    target_swift_type =
        target_swift_type.transformRec([](swift::TypeBase *type)
            -> std::optional<swift::Type> {
          if (auto *dynamic_self = llvm::dyn_cast<swift::DynamicSelfType>(type))
            return dynamic_self->getSelfType();
          return std::nullopt;
        });

    // Thicken generic metatypes. Once substituted, they should always
    // be thick. TypeRef::subst() does the same transformation.
    target_swift_type =
        target_swift_type.transformRec([](swift::TypeBase *type)
            -> std::optional<swift::Type> {
          const auto thin = swift::MetatypeRepresentation::Thin;
          const auto thick = swift::MetatypeRepresentation::Thick;
          if (auto *metatype = swift::dyn_cast<swift::AnyMetatypeType>(type)) {
            if (metatype->hasRepresentation() &&
                metatype->getRepresentation() == thin &&
                metatype->getInstanceType()->hasTypeParameter()) {
              return swift::Type(swift::MetatypeType::get(
                  metatype->getInstanceType(), thick));
            }
          }
          return std::nullopt;
        });

    while (target_swift_type->hasOpaqueArchetype()) {
      auto old_type = target_swift_type;
      target_swift_type = target_swift_type.subst(
          [&](swift::SubstitutableType *type) -> swift::Type {
            auto opaque_type =
                llvm::dyn_cast<swift::OpaqueTypeArchetypeType>(type);
            if (!opaque_type || !opaque_type->getInterfaceType()
                                     ->is<swift::GenericTypeParamType>())
              return type;

            // Try to find the symbol for the opaque type descriptor in the
            // process.
            auto mangled_name = ConstString(
                mangler.mangleOpaqueTypeDescriptor(opaque_type->getDecl()));

            SymbolContextList found;
            target.GetImages().FindSymbolsWithNameAndType(
                mangled_name, eSymbolTypeData, found);

            if (found.GetSize() == 0)
              return type;

            swift::Type result_type;

            for (unsigned i = 0, e = found.GetSize(); i < e; ++i) {
              SymbolContext found_sc;
              if (!found.GetContextAtIndex(i, found_sc))
                continue;

              // See if the symbol has an address.
              if (!found_sc.symbol)
                continue;

              auto addr = found_sc.symbol->GetAddress().GetLoadAddress(&target);
              if (!addr || addr == LLDB_INVALID_ADDRESS)
                continue;

              // Ask RemoteAST to get the underlying type out of the
              // descriptor.
              auto &remote_ast = GetRemoteASTContext(*swift_ast_ctx);
              auto genericParam = opaque_type->getInterfaceType()
                                      ->getAs<swift::GenericTypeParamType>();
              auto underlying_type_result =
                  remote_ast.getUnderlyingTypeForOpaqueType(
                      swift::remote::RemoteAddress(
                          addr,
                          swift::remote::RemoteAddress::DefaultAddressSpace),
                      opaque_type->getSubstitutions(),
                      genericParam->getIndex());

              if (!underlying_type_result)
                continue;

              // If we haven't yet gotten an underlying type, use this as our
              // possible result.
              if (!result_type) {
                result_type = underlying_type_result.getValue();
              }
              // If we have two possibilities, they should match.
              else if (!result_type->isEqual(
                           underlying_type_result.getValue())) {
                return type;
              }
            }

            if (!result_type)
              return type;

            return result_type;
          },
          swift::LookUpConformanceInModule(),
          swift::SubstFlags::DesugarMemberTypes |
              swift::SubstFlags::SubstituteOpaqueArchetypes);

      // Stop if we've reached a fixpoint where we can't further resolve
      // opaque types.
      if (old_type->isEqual(target_swift_type))
        break;
    }

    target_swift_type = target_swift_type.subst(
        [this, &stack_frame,
         swift_ast_ctx](swift::SubstitutableType *type) -> swift::Type {
          StreamString type_name;
          if (!SwiftLanguageRuntime::GetAbstractTypeName(type_name, type))
            return type;
          CompilerType concrete_type = this->GetConcreteType(
              &stack_frame, ConstString(type_name.GetString()));
          Status import_error;
          CompilerType target_concrete_type =
              swift_ast_ctx->ImportType(concrete_type, import_error);

          if (!target_concrete_type.IsValid())
            return type;
          return llvm::expectedToStdOptional(
                     swift_ast_ctx->GetSwiftType(target_concrete_type))
              .value_or(swift::Type());
        },
        swift::LookUpConformanceInModule(),
        swift::SubstFlags::DesugarMemberTypes);
    assert(target_swift_type);

    return ToCompilerType({target_swift_type.getPointer()});
  }
  return base_type;
}

SwiftLanguageRuntime::MetadataPromise::MetadataPromise(
    ValueObject &for_object, SwiftLanguageRuntime &runtime,
    lldb::addr_t location)
    : m_for_object_sp(for_object.GetSP()), m_swift_runtime(runtime),
      m_metadata_location(location) {}

CompilerType SwiftLanguageRuntime::MetadataPromise::FulfillTypePromise(
    const SymbolContext &sc, Status *error) {
  if (error)
    error->Clear();

  Log *log(GetLog(LLDBLog::Types));

  if (log)
    log->Printf("[MetadataPromise] asked to fulfill type promise at location "
                "0x%" PRIx64,
                m_metadata_location);

  if (m_compiler_type.has_value())
    return m_compiler_type.value();

  auto scratch_ctx = TypeSystemSwiftTypeRefForExpressions::GetForTarget(
      m_for_object_sp->GetTargetSP());
  if (!scratch_ctx) {
    if (error)
      *error = Status::FromErrorString("couldn't get Swift scratch context");
    return CompilerType();
  }
  SwiftASTContextSP swift_ast_ctx = scratch_ctx->GetSwiftASTContext(sc);
  if (!swift_ast_ctx) {
    if (error)
      *error = Status::FromErrorString("couldn't get Swift scratch context");
    return CompilerType();
  }
  auto &remote_ast = m_swift_runtime.GetRemoteASTContext(*swift_ast_ctx);
  swift::remoteAST::Result<swift::Type> result =
      remote_ast.getTypeForRemoteTypeMetadata(swift::remote::RemoteAddress(
          m_metadata_location,
          swift::remote::RemoteAddress::DefaultAddressSpace));

  if (result) {
    m_compiler_type = {swift_ast_ctx->weak_from_this(),
                       result.getValue().getPointer()};
    if (log)
      log->Printf("[MetadataPromise] result is type %s",
                  m_compiler_type->GetTypeName().AsCString());
    return m_compiler_type.value();
  } else {
    const auto &failure = result.getFailure();
    if (error)
      *error = Status::FromErrorStringWithFormatv(
          "error in resolving type: {0}", failure.render());
    if (log)
      log->Printf("[MetadataPromise] failure: %s", failure.render().c_str());
    return (m_compiler_type = CompilerType()).value();
  }
}

SwiftLanguageRuntime::MetadataPromiseSP
SwiftLanguageRuntime::GetMetadataPromise(const SymbolContext &sc,
                                         lldb::addr_t addr,
                                         ValueObject &for_object) {
  auto scratch_ctx = TypeSystemSwiftTypeRefForExpressions::GetForTarget(
      for_object.GetTargetSP());
  if (!scratch_ctx)
    return nullptr;
  SwiftASTContextSP swift_ast_ctx = scratch_ctx->GetSwiftASTContext(sc);
  if (!swift_ast_ctx)
    return nullptr;
  if (swift_ast_ctx->HasFatalErrors())
    return nullptr;
  if (addr == 0 || addr == LLDB_INVALID_ADDRESS)
    return nullptr;

  ThreadSafeASTContext ast_ctx = swift_ast_ctx->GetASTContext();
  auto key = std::make_pair(*ast_ctx, addr);
  auto iter = m_promises_map.find(key);
  if (iter != m_promises_map.end())
    return iter->second;

  SwiftLanguageRuntime::MetadataPromiseSP promise_sp(
      new SwiftLanguageRuntime::MetadataPromise(for_object, *this, addr));
  m_promises_map.insert({key, promise_sp});
  return promise_sp;
}

SwiftLanguageRuntime::MetadataPromiseSP
SwiftLanguageRuntime::GetPromiseForTypeNameAndFrame(const char *type_name,
                                                    StackFrame *frame) {
  if (!frame || !type_name || !type_name[0])
    return nullptr;

  StreamString type_metadata_ptr_var_name;
  type_metadata_ptr_var_name.Printf("$%s", type_name);
  VariableList *var_list = frame->GetVariableList(false, nullptr);
  if (!var_list)
    return nullptr;

  VariableSP var_sp(var_list->FindVariable(
      ConstString(type_metadata_ptr_var_name.GetData())));
  if (!var_sp)
    return nullptr;

  ValueObjectSP metadata_ptr_var_sp(
      frame->GetValueObjectForFrameVariable(var_sp, lldb::eNoDynamicValues));
  if (!metadata_ptr_var_sp ||
      metadata_ptr_var_sp->UpdateValueIfNeeded() == false)
    return nullptr;

  lldb::addr_t metadata_location(metadata_ptr_var_sp->GetValueAsUnsigned(0));
  if (metadata_location == 0 || metadata_location == LLDB_INVALID_ADDRESS)
    return nullptr;
  const SymbolContext &sc = frame->GetSymbolContext(eSymbolContextFunction);
  return GetMetadataPromise(sc, metadata_location, *metadata_ptr_var_sp);
}

} // namespace lldb_private
