#' Database versions of the nycflights13 data
#'
#' These functions cache the data from the `nycflights13` database in
#' a local database, for use in examples and vignettes. Indexes are created
#' to making joining tables on natural keys efficient.
#'
#' @keywords internal
#' @name nycflights13
NULL

#' @export
#' @rdname nycflights13
#' @param path location of SQLite database file
nycflights13_sqlite <- function(path = NULL) {
  cache_computation("nycflights_sqlite", {
    path <- db_location(path, "nycflights13.sqlite")
    message("Caching nycflights db at ", path)
    src <- src_sqlite(path, create = TRUE)
    copy_nycflights13(src)
  })
}

#' @export
#' @rdname nycflights13
#' @param dbname,... Arguments passed on to [src_postgres()]
nycflights13_postgres <- function(dbname = "nycflights13", ...) {
  cache_computation("nycflights_postgres", {
    message("Caching nycflights db in postgresql db ", dbname)
    copy_nycflights13(src_postgres(dbname, ...))
  })
}

#' @rdname nycflights13
#' @export
has_nycflights13 <- function(type = c("sqlite", "postgresql"), ...) {
  if (!requireNamespace("nycflights13", quietly = TRUE)) return(FALSE)

  type <- match.arg(type)

  succeeds(switch(
    type,
    sqlite = nycflights13_sqlite(...), quiet = TRUE,
    postgres = nycflights13_postgres(...), quiet = TRUE
  ))
}


#' @export
#' @rdname nycflights13
copy_nycflights13 <- function(src, ...) {
  all <- utils::data(package = "nycflights13")$results[, 3]
  unique_index <- list(
    airlines = list("carrier"),
    planes =   list("tailnum")
  )
  index <- list(
    airports = list("faa"),
    flights =  list(c("year", "month", "day"), "carrier", "tailnum", "origin", "dest"),
    weather =  list(c("year", "month", "day"), "origin")
  )

  tables <- setdiff(all, src_tbls(src))

  # Create missing tables
  for (table in tables) {
    df <- getExportedValue("nycflights13", table)
    message("Creating table: ", table)

    copy_to(
      src, df, table,
      unique_indexes = unique_index[[table]],
      indexes = index[[table]],
      temporary = FALSE
    )
  }
  src
}
