#! /bin/sh
#
#	recite - english text speech synthesizer
#	Copyright (C) 1993 Peter Miller.
#	All rights reserved.
#
#	This program is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; either version 1, or (at your option)
#	any later version.
#
#	This program is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with this program; if not, write to the Free Software
#	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# MANIFEST: Test the phonemes to frames translation
#

work=/tmp/$$
PAGER=cat
export PAGER

fail()
{
	set +x
	echo FAILED test of phonemes to frames translation 1>&2
	rm -rf $work
	exit 1
}
pass()
{
	set +x
	rm -rf $work
	exit 0
}
trap "fail" 1 2 3 15

here=`pwd`
if test $? -ne 0 ; then exit 1; fi
mkdir $work
if test $? -ne 0 ; then exit 1; fi
cd $work
if test $? -ne 0 ; then fail; fi

#
# first
#
cat > test.in << 'fubar'
DHAX kwIHk brOWn fAAks jUWmps OWvER DHAX lEYzIH dAAg

fubar
if test $? -ne 0 ; then fail; fi
cat > test.ok.Z.uue << 'fubar'
begin 640 test.ok.Z
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#I ( 
 
end
fubar
if test $? -ne 0 ; then fail; fi
uudecode test.ok.Z.uue
if test $? -ne 0 ; then fail; fi
uncompress test.ok
if test $? -ne 0 ; then fail; fi
$here/bin/recite -rp -wk < test.in > test.out
if test $? -ne 0 ; then fail; fi
diff test.ok test.out
if test $? -ne 0 ; then fail; fi
rm test.in test.out test.ok
if test $? -ne 0 ; then fail; fi

#
# second
#
cat > test.in << 'fubar'
nAW IHz DHAX tAYm fAOr AOl gUHd mEHn tUW kAHm tUW DHAX EYd AXv DHEHr pAArtIH

fubar
if test $? -ne 0 ; then fail; fi
cat > test.ok.Z.uue << 'fubar'
begin 640 test.ok.Z
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J\ S&P#LX!#M)'YR!VQ0/KK]/F 0#801,9*AP%#["3G@(>^ 7K('!L-H!
 
end
fubar
if test $? -ne 0 ; then fail; fi
uudecode test.ok.Z.uue
if test $? -ne 0 ; then fail; fi
uncompress test.ok
if test $? -ne 0 ; then fail; fi
$here/bin/recite -rp -wk < test.in > test.out
if test $? -ne 0 ; then fail; fi
diff test.ok test.out
if test $? -ne 0 ; then fail; fi
rm test.in test.out test.ok
if test $? -ne 0 ; then fail; fi

#
# third
#
cat > test.in << 'fubar'
IHt IHz jIYnERAXlIY bIHlIYIYvd DHAEt AOl dAAktERz AAr tERkIYs pIYrIYAAd
DHIHs IHz nAAt kAOrrEHkt pIYrIYAAd
mEHnIY dAAktERz AAr kwAEks pIYrIYAAd
fubar
if test $? -ne 0 ; then fail; fi
cat > test.ok.Z.uue << 'fubar'
begin 640 test.ok.Z
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*.7I*6\=G[ZD  ,GK
 
end
fubar
if test $? -ne 0 ; then fail; fi
uudecode test.ok.Z.uue
if test $? -ne 0 ; then fail; fi
uncompress test.ok
if test $? -ne 0 ; then fail; fi
$here/bin/recite -rp -wk < test.in > test.out
if test $? -ne 0 ; then fail; fi
diff test.ok test.out
if test $? -ne 0 ; then fail; fi

#
# Only definite negatives are possible.
# The functionality exercised by this test appears to work,
# no other guarantees are made.
#
pass
