///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "2W.h"
#include "rheolef/ublas_matrix_range.h"
using namespace rheolef;
using namespace std;
using namespace ublas;

// vorticity tensor
// 
// -------------------
// bidimensionnal case
// -------------------
//  2W(u) = ([0,-w],[w,0])
//  where w = du1/dx0 - du0/dx1
//
//          [ 0    -w ]
//  2W(u) = [         ]
//          [ w    0  ]
//
// ordered by column: tau = 2W
//
//                [   u0   u1 ]
//  
//  [ tau_00 ]    [   0     0  ]
//  [ tau_01 ]    [  dx1  -dx0 ]
//  [ tau_10 ] =  [ -dx1   dx0 ]
//  [ tau_11 ]    [   0     0  ]
//
// --------------------
// tridimensionnal case
// --------------------
//
//       [         2W_01   2W_02 ]
//  2W = [ 2W_10           2W_12 ]
//       [ 2W_20   2W_21         ]
//
// ordered by column: tau = 2W
//
//                [   u0    u1    u2 ]
//
//  [ tau_00 ] =  [                   ]
//  [ tau_01 ] =  [  dx1  -dx0        ]
//  [ tau_10 ] =  [ -dx1   dx0        ]
//  [ tau_11 ] =  [                   ]
//  [ tau_02 ] =  [  dx2         -dx0 ]
//  [ tau_12 ] =  [        dx2   -dx1 ]
//  [ tau_20 ] =  [ -dx2          dx0 ]
//  [ tau_21 ] =  [       -dx2    dx1 ]
//  [ tau_22 ] =  [                   ]
//
// ------------
// axisymmetric
// ------------
//
//  2W(u) = ([0,-w],[w,0])
//  where w = du1/dx0 - du0/dx1
//
//          [ 0  -w   0 ]
//  2W(u) = [ w  0    0 ]
//          [ 0  0    0 ]
//
// ordered by column:
//
//                [   u0    u1 ]
//
//  [ tau_00 ]    [   0     0  ]
//  [ tau_01 ]    [  dx1  -dx0 ]
//  [ tau_10 ] =  [ -dx1   dx0 ]
//  [ tau_11 ]    [   0     0  ]
//  [ tau_22 ]    [   0     0  ]
//
void
_2W::operator() (const geo_element& K, matrix<Float>& dd) const
{
  switch (K.dimension()) {
    case 2: {
      matrix<Float> dx0, dx1;
      d_dx (K, dx0, 0);
      d_dx (K, dx1, 1);
      size_type ni = dx0.size1();
      size_type nj = dx0.size2();
      size_type n_comp;
      if (coordinate_system_type() == fem_helper::cartesian) n_comp=4;
      else n_comp=5; // rz or zr axisymmetry
      dd.resize (n_comp*ni, 2*nj);
      dd.clear();
      //  [ 2W_00 ]    [   0     0  ]
      //  [ 2W_01 ]    [  dx1  -dx0 ]
      //  [ 2W_10 ] =  [ -dx1   dx0 ]
      //  [ 2W_11 ]    [   0     0  ]
      mr_set (dd, range(ni,2*ni),    range(0,nj),        dx1);
      mr_set (dd, range(ni,2*ni),    range(nj,2*nj),    -dx0);
      mr_set (dd, range(2*ni,3*ni),  range(0,nj),       -dx1);
      mr_set (dd, range(2*ni,3*ni),  range(nj,2*nj),     dx0);
      break;
    }
    case 3: {
      matrix<Float> dx0, dx1, dx2;
      d_dx (K, dx0, 0);
      d_dx (K, dx1, 1);
      d_dx (K, dx2, 2);
      size_type ni = dx0.size1();
      size_type nj = dx0.size2();
      dd.resize (9*ni, 3*nj);
      dd.clear();
      //  [ 2W_00 ] =  [                   ]
      //  [ 2W_01 ] =  [  dx1  -dx0        ]
      //  [ 2W_10 ] =  [ -dx1   dx0        ]
      //  [ 2W_11 ] =  [                   ]
      mr_set (dd, range(ni,2*ni),    range(0,nj),        dx1);
      mr_set (dd, range(ni,2*ni),    range(nj,2*nj),    -dx0);
      mr_set (dd, range(2*ni,3*ni),  range(0,nj),       -dx1);
      mr_set (dd, range(2*ni,3*ni),  range(nj,2*nj),     dx0);
      //  [ 2W_02 ] =  [  dx2         -dx0 ]
      mr_set (dd, range(4*ni,5*ni),  range(0,nj),        dx2);
      mr_set (dd, range(4*ni,5*ni),  range(2*nj,3*nj),  -dx0);
      //  [ 2W_12 ] =  [        dx2   -dx1 ]
      mr_set (dd, range(5*ni,6*ni),  range(nj,2*nj),     dx2);
      mr_set (dd, range(5*ni,6*ni),  range(2*nj,3*nj),  -dx1);
      //  [ 2W_20 ] =  [ -dx2          dx0 ]
      mr_set (dd, range(6*ni,7*ni),  range(0,nj),       -dx2);
      mr_set (dd, range(6*ni,7*ni),  range(2*nj,3*nj),   dx0);
      //  [ 2W_21 ] =  [       -dx2    dx1 ]
      //  [ 2W_22 ] =  [                   ]
      mr_set (dd, range(7*ni,8*ni),  range(nj,2*nj),    -dx2);
      mr_set (dd, range(7*ni,8*ni),  range(2*nj,3*nj),   dx1);
      break;
    }
    default: {
      fatal_macro ("form `2W' not supported in " << K.dimension() << "d geometry");
    }
  }
}
_2W::size_type
_2W::n_derivative() const
{
  return 1;
}
