/* lkcapi_dh_glue.c -- glue logic to register dh and ffdhe wolfCrypt
 * implementations with the Linux Kernel Cryptosystem
 *
 * Copyright (C) 2006-2025 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */

/* included by linuxkm/lkcapi_glue.c */
#ifndef WC_SKIP_INCLUDED_C_FILES

#ifndef LINUXKM_LKCAPI_REGISTER
    #error lkcapi_dh_glue.c included in non-LINUXKM_LKCAPI_REGISTER project.
#endif

#if (defined(LINUXKM_LKCAPI_REGISTER_ALL) || \
     (defined(LINUXKM_LKCAPI_REGISTER_ALL_KCONFIG) && defined(CONFIG_CRYPTO_DH))) && \
    !defined(LINUXKM_LKCAPI_DONT_REGISTER_DH) &&  \
    !defined(LINUXKM_LKCAPI_REGISTER_DH)
    #define LINUXKM_LKCAPI_REGISTER_DH
    #define LINUXKM_DH
#endif

#if defined(LINUXKM_LKCAPI_REGISTER_DH) && \
    !defined(WOLFSSL_DH_EXTRA)
     /* not supported without WOLFSSL_DH_EXTRA */
    #undef LINUXKM_LKCAPI_REGISTER_DH

    #if defined(LINUXKM_LKCAPI_REGISTER_ALL_KCONFIG) && defined(CONFIG_CRYPTO_DH)
        #error Config conflict: missing features force off LINUXKM_LKCAPI_REGISTER_DH.
    #endif
#endif /* LINUXKM_LKCAPI_REGISTER_DH */

#if defined (LINUXKM_LKCAPI_REGISTER_DH) && defined(CONFIG_CRYPTO_FIPS) && \
    defined(CONFIG_CRYPTO_MANAGER)
        /*
         * note: normal dh not fips_allowed in kernel crypto/testmgr.c,
         * and will not pass the tests.
         */
        #undef LINUXKM_DH
#endif /* LINUXKM_LKCAPI_REGISTER_DH */

#ifdef NO_DH
    #undef LINUXKM_LKCAPI_REGISTER_DH
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 18, 0)
    /* Support for FFDHE was added in kernel 5.18, and generic DH support
     * pre-5.18 used a different binary format for the secret (an additional
     * slot for q).
     *
     * LTS backports of FFDHE go as far back as 4.14.236, using the pre-5.18
     * binary layout, but other backports, e.g. RHEL 9.5 on kernel
     * 5.14.0-503.40.1, have the 5.18+ layout.  Best to disable on all pre-5.18
     * and triage as/if necessary.
     */
    #undef LINUXKM_LKCAPI_REGISTER_DH
#endif

#if defined(LINUXKM_LKCAPI_REGISTER_ALL_KCONFIG) && \
    (defined(CONFIG_CRYPTO_DH) || defined(CONFIG_CRYPTO_DH_RFC7919_GROUPS)) && \
    !defined(LINUXKM_LKCAPI_REGISTER_DH)
    #error Config conflict: target kernel has CONFIG_CRYPTO_DH and/or \
        _DH_RFC7919_GROUPS, but module is missing LINUXKM_LKCAPI_REGISTER_DH.
#endif

#if defined(LINUXKM_LKCAPI_REGISTER_DH)

#include <wolfssl/wolfcrypt/asn.h>
#include <wolfssl/wolfcrypt/dh.h>
#include <crypto/dh.h>

#define WOLFKM_DH_NAME    ("dh")
#define WOLFKM_DH_DRIVER  ("dh" WOLFKM_DRIVER_FIPS \
                           "-wolfcrypt")

#ifdef HAVE_FFDHE_2048
    #define WOLFKM_FFDHE2048_NAME   ("ffdhe2048(dh)")
    #define WOLFKM_FFDHE2048_DRIVER ("ffdhe2048" WOLFKM_DRIVER_FIPS \
                                     "-wolfcrypt")
#endif /* HAVE_FFDHE_2048 */

#ifdef HAVE_FFDHE_3072
    #define WOLFKM_FFDHE3072_NAME   ("ffdhe3072(dh)")
    #define WOLFKM_FFDHE3072_DRIVER ("ffdhe3072" WOLFKM_DRIVER_FIPS \
                                     "-wolfcrypt")
#endif /* HAVE_FFDHE_3072 */

#ifdef HAVE_FFDHE_4096
    #define WOLFKM_FFDHE4096_NAME   ("ffdhe4096(dh)")
    #define WOLFKM_FFDHE4096_DRIVER ("ffdhe4096" WOLFKM_DRIVER_FIPS \
                                     "-wolfcrypt")
#endif /* HAVE_FFDHE_4096 */

#ifdef HAVE_FFDHE_6144
    #define WOLFKM_FFDHE6144_NAME   ("ffdhe6144(dh)")
    #define WOLFKM_FFDHE6144_DRIVER ("ffdhe6144" WOLFKM_DRIVER_FIPS \
                                     "-wolfcrypt")
#endif /* HAVE_FFDHE_6144 */

#ifdef HAVE_FFDHE_8192
    #define WOLFKM_FFDHE8192_NAME   ("ffdhe8192(dh)")
    #define WOLFKM_FFDHE8192_DRIVER ("ffdhe8192" WOLFKM_DRIVER_FIPS \
                                     "-wolfcrypt")
#endif /* HAVE_FFDHE_8192 */

static int linuxkm_test_kpp_driver(const char * driver,
                                   const byte * b_pub,
                                   const byte * expected_a_pub,
                                   word32 pub_len,
                                   const byte * secret,
                                   word32 secret_len,
                                   const byte * shared_secret,
                                   word32 shared_s_len);

#ifdef LINUXKM_DH
static int dh_loaded = 0;
#endif /* LINUXKM_DH */
#ifdef HAVE_FFDHE_2048
static int ffdhe2048_loaded = 0;
#endif /* HAVE_FFDHE_2048 */
#ifdef HAVE_FFDHE_3072
static int ffdhe3072_loaded = 0;
#endif /* HAVE_FFDHE_3072 */
#ifdef HAVE_FFDHE_4096
static int ffdhe4096_loaded = 0;
#endif /* HAVE_FFDHE_4096 */
#ifdef HAVE_FFDHE_6144
static int ffdhe6144_loaded = 0;
#endif /* HAVE_FFDHE_6144 */
#ifdef HAVE_FFDHE_8192
static int ffdhe8192_loaded = 0;
#endif /* HAVE_FFDHE_8192 */

struct km_dh_ctx {
    WC_RNG   rng; /* needed for keypair gen and timing resistance*/
    DhKey *  key;
    byte *   priv_key;
    byte *   pub_key;
    int      name; /* the finite-field group name */
    word32   nbits;
    word32   priv_len;
    word32   pub_len;
    byte     has_pub_key;
};

/* shared misc functions */
static int          km_dh_decode_secret(const u8 * buf, unsigned int len,
                                        struct dh * params);
static int          km_dh_alloc_keys(struct km_dh_ctx * ctx);
static int          km_dh_reset_ctx(struct km_dh_ctx * ctx);
static int          km_ffdhe_init(struct crypto_kpp *tfm, int name,
                                  word32 nbits);

/* shared callbacks */
static unsigned int km_dh_max_size(struct crypto_kpp *tfm);
static int          km_dh_gen_pub(struct kpp_request *req);
static int          km_dh_compute_shared_secret(struct kpp_request *req);
static void         km_dh_exit(struct crypto_kpp *tfm);

/* alg specific callbacks */
#ifdef LINUXKM_DH
static int          km_dh_init(struct crypto_kpp *tfm);
static int          km_dh_set_secret(struct crypto_kpp *tfm, const void *buf,
                                     unsigned int len);
#endif /* LINUXKM_DH */
static int          km_ffdhe_set_secret(struct crypto_kpp *tfm,
                                        const void *buf, unsigned int len);
#ifdef HAVE_FFDHE_2048
static int          km_ffdhe2048_init(struct crypto_kpp *tfm);
#endif /* HAVE_FFDHE_2048 */
#ifdef HAVE_FFDHE_3072
static int          km_ffdhe3072_init(struct crypto_kpp *tfm);
#endif /* HAVE_FFDHE_3072 */
#ifdef HAVE_FFDHE_4096
static int          km_ffdhe4096_init(struct crypto_kpp *tfm);
#endif /* HAVE_FFDHE_4096 */
#ifdef HAVE_FFDHE_6144
static int          km_ffdhe6144_init(struct crypto_kpp *tfm);
#endif /* HAVE_FFDHE_6144 */
#ifdef HAVE_FFDHE_8192
static int          km_ffdhe8192_init(struct crypto_kpp *tfm);
#endif /* HAVE_FFDHE_8192 */

#ifdef LINUXKM_DH
static struct kpp_alg dh = {
    .base.cra_name         = WOLFKM_DH_NAME,
    .base.cra_driver_name  = WOLFKM_DH_DRIVER,
    .base.cra_priority     = WOLFSSL_LINUXKM_LKCAPI_PRIORITY,
    .base.cra_module       = THIS_MODULE,
    .base.cra_ctxsize      = sizeof(struct km_dh_ctx),
    .set_secret            = km_dh_set_secret,
    .generate_public_key   = km_dh_gen_pub,
    .compute_shared_secret = km_dh_compute_shared_secret,
    .max_size              = km_dh_max_size,
    .init                  = km_dh_init,
    .exit                  = km_dh_exit,
};
#endif /* LINUXKM_DH */

#ifdef HAVE_FFDHE_2048
static struct kpp_alg ffdhe2048 = {
    .base.cra_name         = WOLFKM_FFDHE2048_NAME,
    .base.cra_driver_name  = WOLFKM_FFDHE2048_DRIVER,
    .base.cra_priority     = WOLFSSL_LINUXKM_LKCAPI_PRIORITY,
    .base.cra_module       = THIS_MODULE,
    .base.cra_ctxsize      = sizeof(struct km_dh_ctx),
    .set_secret            = km_ffdhe_set_secret,
    .generate_public_key   = km_dh_gen_pub,
    .compute_shared_secret = km_dh_compute_shared_secret,
    .max_size              = km_dh_max_size,
    .init                  = km_ffdhe2048_init,
    .exit                  = km_dh_exit,
};
#endif /* HAVE_FFDHE_2048 */

#ifdef HAVE_FFDHE_3072
static struct kpp_alg ffdhe3072 = {
    .base.cra_name         = WOLFKM_FFDHE3072_NAME,
    .base.cra_driver_name  = WOLFKM_FFDHE3072_DRIVER,
    .base.cra_priority     = WOLFSSL_LINUXKM_LKCAPI_PRIORITY,
    .base.cra_module       = THIS_MODULE,
    .base.cra_ctxsize      = sizeof(struct km_dh_ctx),
    .set_secret            = km_ffdhe_set_secret,
    .generate_public_key   = km_dh_gen_pub,
    .compute_shared_secret = km_dh_compute_shared_secret,
    .max_size              = km_dh_max_size,
    .init                  = km_ffdhe3072_init,
    .exit                  = km_dh_exit,
};
#endif /* HAVE_FFDHE_3072 */

#ifdef HAVE_FFDHE_4096
static struct kpp_alg ffdhe4096 = {
    .base.cra_name         = WOLFKM_FFDHE4096_NAME,
    .base.cra_driver_name  = WOLFKM_FFDHE4096_DRIVER,
    .base.cra_priority     = WOLFSSL_LINUXKM_LKCAPI_PRIORITY,
    .base.cra_module       = THIS_MODULE,
    .base.cra_ctxsize      = sizeof(struct km_dh_ctx),
    .set_secret            = km_ffdhe_set_secret,
    .generate_public_key   = km_dh_gen_pub,
    .compute_shared_secret = km_dh_compute_shared_secret,
    .max_size              = km_dh_max_size,
    .init                  = km_ffdhe4096_init,
    .exit                  = km_dh_exit,
};
#endif /* HAVE_FFDHE_4096 */

#ifdef HAVE_FFDHE_6144
static struct kpp_alg ffdhe6144 = {
    .base.cra_name         = WOLFKM_FFDHE6144_NAME,
    .base.cra_driver_name  = WOLFKM_FFDHE6144_DRIVER,
    .base.cra_priority     = WOLFSSL_LINUXKM_LKCAPI_PRIORITY,
    .base.cra_module       = THIS_MODULE,
    .base.cra_ctxsize      = sizeof(struct km_dh_ctx),
    .set_secret            = km_ffdhe_set_secret,
    .generate_public_key   = km_dh_gen_pub,
    .compute_shared_secret = km_dh_compute_shared_secret,
    .max_size              = km_dh_max_size,
    .init                  = km_ffdhe6144_init,
    .exit                  = km_dh_exit,
};
#endif /* HAVE_FFDHE_6144 */

#ifdef HAVE_FFDHE_8192
static struct kpp_alg ffdhe8192 = {
    .base.cra_name         = WOLFKM_FFDHE8192_NAME,
    .base.cra_driver_name  = WOLFKM_FFDHE8192_DRIVER,
    .base.cra_priority     = WOLFSSL_LINUXKM_LKCAPI_PRIORITY,
    .base.cra_module       = THIS_MODULE,
    .base.cra_ctxsize      = sizeof(struct km_dh_ctx),
    .set_secret            = km_ffdhe_set_secret,
    .generate_public_key   = km_dh_gen_pub,
    .compute_shared_secret = km_dh_compute_shared_secret,
    .max_size              = km_dh_max_size,
    .init                  = km_ffdhe8192_init,
    .exit                  = km_dh_exit,
};
#endif /* HAVE_FFDHE_8192 */

/* The DH secret is passed in this format:
 *    ______________________________________________________________________
 *   | secret hdr          | key_size | p_size | g_size |  key  |  p  |  g  |
 *   | (struct kpp_secret) | (int)    | (int)  | (int)  |       |     |     |
 *    ----------------------------------------------------------------------
 *
 * normal dh:
 *   - the key, p, g must all be provided.
 *
 * ffdhe:
 *   - p, g must be empty.
 *   - the key is optional.
 *
 * The key_size, p_size, g_size fields are always provided, even if zero.
 * */
#define DH_KPP_SECRET_MIN_SIZE (sizeof(struct kpp_secret) + 3 * sizeof(int))

static int km_dh_decode_secret(const u8 * buf, unsigned int len,
                               struct dh * params)
{
    struct kpp_secret secret;
    const u8 *        ptr = NULL;
    size_t            expected_len = 0;

    if (unlikely(!buf || len < DH_KPP_SECRET_MIN_SIZE || !params)) {
        return -EINVAL;
    }

    /* the type of secret should be the first byte. */
    ptr = buf;
    memcpy(&secret, ptr, sizeof(secret));
    ptr += sizeof(secret);
    if (secret.type != CRYPTO_KPP_SECRET_TYPE_DH) {
        return -EINVAL;
    }

    /* all three of these fields will be present */
    memcpy(&params->key_size, ptr, sizeof(params->key_size));
    ptr += sizeof(params->key_size);
    memcpy(&params->p_size, ptr, sizeof(params->p_size));
    ptr += sizeof(params->p_size);
    memcpy(&params->g_size, ptr, sizeof(params->g_size));
    ptr += sizeof(params->g_size);

    /* Calculate expected len based on provided 3 fields. Verify
     * we got expected data. */
    expected_len = DH_KPP_SECRET_MIN_SIZE + params->key_size + params->p_size
                 + params->g_size;

    if (secret.len != expected_len) {
        #ifdef WOLFKM_DEBUG_DH
        pr_err("%s: km_dh_decode_secret: got %d, expected %zu",
               WOLFKM_DH_DRIVER, secret.len, expected_len);
        #endif /* WOLFKM_DEBUG_DH */
        return -EINVAL;
    }

    /* The rest of the fields are optional depending on how much data
     * was passed, and whether it's ffdhe or dh. */
    if (params->key_size) {
        params->key = (void *)ptr;

        if (params->p_size && params->g_size) {
            ptr += params->key_size;
            params->p = (void *)ptr;
            params->g = (void *)(ptr + params->p_size);
        }
    }

    return 0;
}

static inline void km_dh_clear_keys(struct km_dh_ctx * ctx)
{
    if (ctx->priv_key) {
        ForceZero(ctx->priv_key, ctx->priv_len);
        free(ctx->priv_key);
        ctx->priv_key = NULL;
        ctx->priv_len = 0;
    }

    if (ctx->pub_key) {
        free(ctx->pub_key);
        ctx->pub_key = NULL;
        ctx->pub_len = 0;
    }

    return;
}

static int km_dh_alloc_keys(struct km_dh_ctx * ctx)
{
    int err = 0;

    /* allocate priv and pub key arrays. */
    ctx->priv_len = ctx->nbits / WOLFSSL_BIT_SIZE;
    ctx->pub_len = ctx->nbits / WOLFSSL_BIT_SIZE;

    ctx->priv_key = malloc(ctx->priv_len);
    if (!ctx->priv_key) {
        err = -ENOMEM;
        goto alloc_keys_end;
    }

    memset(ctx->priv_key, 0, ctx->priv_len);

    ctx->pub_key = malloc(ctx->pub_len);
    if (!ctx->pub_key) {
        err = -ENOMEM;
        goto alloc_keys_end;
    }

    memset(ctx->pub_key, 0, ctx->pub_len);

alloc_keys_end:
    if (err) {
        #ifdef WOLFKM_DEBUG_DH
        pr_err("%s: km_dh_alloc_keys failed: %d",
               WOLFKM_DH_DRIVER, err);
        #endif
        km_dh_clear_keys(ctx);
    }

    return err;
}

static int km_dh_reset_ctx(struct km_dh_ctx * ctx)
{
    int err = 0;

    /* reset old dh key. */
    if (ctx->priv_key) {
        wc_FreeDhKey(ctx->key);
        err = wc_InitDhKey(ctx->key);

        if (unlikely(err)) {
            err = -ENOMEM;
            goto reset_ctx_end;
        }

        if (ctx->name) {
            err = wc_DhSetNamedKey(ctx->key, ctx->name);
            if (err) {
                #ifdef WOLFKM_DEBUG_DH
                pr_err("%s: wc_DhSetNamedKey returned: %d\n", WOLFKM_DH_DRIVER,
                       err);
                #endif /* WOLFKM_DEBUG_DH */
                err = -ENOMEM;
                goto reset_ctx_end;
            }
        }
    }

reset_ctx_end:
    /* clear old priv and public key arrays. */
    km_dh_clear_keys(ctx);

    return err;
}

#ifdef LINUXKM_DH
/**
 * Set the secret. Kernel crypto expects secret is passed with
 * struct kpp_secret as header, followed by secret data as payload.
 * See these for more info:
 *  - crypto/dh_helper.c
 *  - include/crypto/kpp.h
 *
 * dh does not accept an empty payload, unlike ffdhe, and ecdh.
 * */
static int km_dh_set_secret(struct crypto_kpp *tfm, const void *buf,
                            unsigned int len)
{
    int                err = -1;
    struct km_dh_ctx * ctx = NULL;
    struct dh          params;

    ctx = kpp_tfm_ctx(tfm);
    memset(&params, 0, sizeof(params));

    if (km_dh_decode_secret(buf, len, &params) < 0) {
        #ifdef WOLFKM_DEBUG_DH
        pr_err("%s: dh_set_secret: decode secret failed: %d",
               WOLFKM_DH_DRIVER, params.key_size);
        #endif /* WOLFKM_DEBUG_DH */
        err = -EINVAL;
        goto dh_secret_end;
    }

    /* the key, p, and g, must all be provided for normal dh. */
    if (!params.key || !params.key_size || !params.p_size || !params.g_size) {
        #ifdef WOLFKM_DEBUG_DH
        pr_err("%s: dh_set_secret: empty params", WOLFKM_DH_DRIVER);
        #endif
        err = -EINVAL;
        goto dh_secret_end;
    }

    if (params.key_size > params.p_size || params.g_size > params.p_size) {
        #ifdef WOLFKM_DEBUG_DH
        pr_err("%s: dh_set_secret: invalid params", WOLFKM_DH_DRIVER);
        #endif
        err = -EINVAL;
        goto dh_secret_end;
    }

    err = km_dh_reset_ctx(ctx);
    if (err) {
        goto dh_secret_end;
    }

    err = km_dh_alloc_keys(ctx);
    if (err) {
        goto dh_secret_end;
    }

    /* set dh params */
    err = wc_DhSetKey(ctx->key, params.p, params.p_size,
                      params.g, params.g_size);

    if (err) {
        #ifdef WOLFKM_DEBUG_DH
        pr_err("%s: wc_DhSetKey failed: %d", WOLFKM_DH_DRIVER, err);
        #endif
        err = -EINVAL;
        goto dh_secret_end;
    }

    /* import private key */
    err = wc_DhImportKeyPair(ctx->key, params.key, params.key_size,
                             NULL, 0);

    if (err) {
        #ifdef WOLFKM_DEBUG_DH
        pr_err("%s: wc_DhImportKeyPair failed: %d", WOLFKM_DH_DRIVER, err);
        #endif
        err = -EINVAL;
        goto dh_secret_end;
    }

    err = wc_DhExportKeyPair(ctx->key, ctx->priv_key, &ctx->priv_len,
                             NULL, NULL);

    if (err) {
        #ifdef WOLFKM_DEBUG_DH
        pr_err("%s: wc_DhExportKeyPair failed: %d\n",
               WOLFKM_DH_DRIVER, err);
        #endif
        err = -EINVAL;
        goto dh_secret_end;
    }

    ctx->has_pub_key = 0;
dh_secret_end:

    #ifdef WOLFKM_DEBUG_DH
    pr_info("info: exiting km_dh_set_secret\n");
    #endif /* WOLFKM_DEBUG_DH */
    return err;
}
#endif /* LINUXKM_DH */

/**
 * Set the secret. Kernel crypto expects secret is passed with
 * struct kpp_secret as header, followed by secret data as payload.
 * See these for more info:
 *  - crypto/dh_helper.c
 *  - include/crypto/kpp.h
 *
 * - ffdhe should not pass p or g.
 * - passing key is optional. If key is not provided, it should
 *   be generated.
 * */
static int km_ffdhe_set_secret(struct crypto_kpp *tfm, const void *buf,
                               unsigned int len)
{
    int                err = -1;
    struct km_dh_ctx * ctx = NULL;
    struct dh          params;

    ctx = kpp_tfm_ctx(tfm);
    memset(&params, 0, sizeof(params));

    /* buf is optional for ffdhe */
    if (buf) {
        err = km_dh_decode_secret((u8*)buf, len, &params);

        if (err) {
            #ifdef WOLFKM_DEBUG_DH
            pr_err("%s: ffdhe_set_secret: decode secret failed: %d",
                   WOLFKM_DH_DRIVER, params.key_size);
            #endif /* WOLFKM_DEBUG_DH */
            err = -EINVAL;
            goto ffdhe_secret_end;
        }

        /* p_size and g_size should be 0 for ffdhe. */
        if (params.p_size || params.g_size) {
            #ifdef WOLFKM_DEBUG_DH
            pr_err("%s: ffdhe_set_secret: unexpected p, g params: %d, %d",
                   WOLFKM_DH_DRIVER, params.p_size, params.g_size);
            #endif /* WOLFKM_DEBUG_DH */
            err = -EINVAL;
            goto ffdhe_secret_end;
        }
    }

    err = km_dh_reset_ctx(ctx);
    if (err) {
        goto ffdhe_secret_end;
    }

    err = km_dh_alloc_keys(ctx);
    if (err) {
        goto ffdhe_secret_end;
    }

    if (!params.key_size) {
        /* generate the ffdhe key pair*/
        #ifdef WOLFKM_DEBUG_DH
        pr_info("ffdhe gen key pair");
        #endif
        PRIVATE_KEY_UNLOCK();
        err = wc_DhGenerateKeyPair(ctx->key, &ctx->rng,
                                   ctx->priv_key, &ctx->priv_len,
                                   ctx->pub_key, &ctx->pub_len);
        PRIVATE_KEY_LOCK();

        if (err) {
            #ifdef WOLFKM_DEBUG_DH
            pr_err("%s: wc_DhGenerateKeyPair failed: %d",
                   WOLFKM_DH_DRIVER, err);
            #endif
            err = -EINVAL;
            goto ffdhe_secret_end;
        }

        /* Pad with leading zeros so pub key is modulus length.
         *
         * The kernel crypto/testmgr.c kpp test assumes same fixed width array
         * for all secrets and pub keys, and doesn't check for updates to
         * req->dst_len if the pub key is shorter because of leading zeros. */
        if (ctx->pub_len < (ctx->nbits / WOLFSSL_BIT_SIZE)) {
            word32 pad_len = ctx->nbits / WOLFSSL_BIT_SIZE - ctx->pub_len;
            #ifdef WOLFKM_DEBUG_DH
            pr_info("info: km_ffdhe_set_secret: pub key padding %d", pad_len);
            #endif

            memmove(ctx->pub_key + pad_len, ctx->pub_key, ctx->pub_len);
            memset(ctx->pub_key, 0, pad_len);

            ctx->pub_len += pad_len;
        }

        /* already has the pub key generated. */
        ctx->has_pub_key = 1;
    }
    else {
        /* import the private key. */
        err = wc_DhImportKeyPair(ctx->key, params.key, params.key_size,
                                 NULL, 0);

        if (err) {
            #ifdef WOLFKM_DEBUG_DH
            pr_err("%s: wc_DhImportKeyPair failed: %d",
                   WOLFKM_DH_DRIVER, err);
            #endif
            err = -EINVAL;
            goto ffdhe_secret_end;
        }

        err = wc_DhExportKeyPair(ctx->key, ctx->priv_key, &ctx->priv_len,
                                 NULL, NULL);

        if (err) {
            #ifdef WOLFKM_DEBUG_DH
            pr_err("%s: wc_DhExportKeyPair failed: %d\n",
                   WOLFKM_DH_DRIVER, err);
            #endif
            err = -EINVAL;
            goto ffdhe_secret_end;
        }

        /* needs pub key generation */
        ctx->has_pub_key = 0;
    }

ffdhe_secret_end:
    if (unlikely(err)) {
        km_dh_reset_ctx(ctx);
    }

    #ifdef WOLFKM_DEBUG_DH
    pr_info("info: exiting km_dh_set_secret\n");
    #endif /* WOLFKM_DEBUG_DH */
    return err;
}

/**
 * Returns modulus size for this dh key.
 *
 * Requires that crypto_kpp_set_secret has been called first.
 * */
static unsigned int km_dh_max_size(struct crypto_kpp *tfm)
{
    struct km_dh_ctx * ctx = NULL;
    unsigned int       max_size = 0;

    ctx = kpp_tfm_ctx(tfm);
    max_size = mp_unsigned_bin_size(&ctx->key->p);

    #ifdef WOLFKM_DEBUG_DH
    pr_info("info: exiting km_dh_max_size: %d\n", max_size);
    #endif /* WOLFKM_DEBUG_DH */
    return max_size;
}

static void km_dh_exit(struct crypto_kpp *tfm)
{
    struct km_dh_ctx * ctx = NULL;

    ctx = kpp_tfm_ctx(tfm);

    if (ctx->key) {
        wc_FreeDhKey(ctx->key);
        free(ctx->key);
        ctx->key = NULL;
    }

    km_dh_clear_keys(ctx);
    wc_FreeRng(&ctx->rng);

    #ifdef WOLFKM_DEBUG_DH
    pr_info("info: exiting km_dh_exit\n");
    #endif /* WOLFKM_DEBUG_DH */
    return;
}

/* Inits the dh tfm with the given ff named group and bit size.
 *
 * if name == 0, inits as dh.
 * if name != 0, inits as ffdhe.
 * */
static int km_ffdhe_init(struct crypto_kpp *tfm, int name, word32 nbits)
{
    struct km_dh_ctx * ctx = NULL;
    int                err = 0;

    ctx = kpp_tfm_ctx(tfm);
    memset(ctx, 0, sizeof(struct km_dh_ctx));
    ctx->name = name;
    ctx->nbits = nbits;

    err = wc_InitRng(&ctx->rng);
    if (err) {
        #ifdef WOLFKM_DEBUG_DH
        pr_err("%s: init rng returned: %d\n", WOLFKM_DH_DRIVER, err);
        #endif /* WOLFKM_DEBUG_DH */
        return -ENOMEM;
    }

    ctx->key = (DhKey *)malloc(sizeof(DhKey));
    if (!ctx->key) {
        return -ENOMEM;
    }

    err = wc_InitDhKey(ctx->key);
    if (err < 0) {
        free(ctx->key);
        ctx->key = NULL;
        return -ENOMEM;
    }

    if (ctx->name) {
        err = wc_DhSetNamedKey(ctx->key, ctx->name);
        if (err) {
            #ifdef WOLFKM_DEBUG_DH
            pr_err("%s: wc_DhSetNamedKey returned: %d\n", WOLFKM_DH_DRIVER,
                   err);
            #endif /* WOLFKM_DEBUG_DH */
            free(ctx->key);
            ctx->key = NULL;
            return -ENOMEM;
        }
    }

    #ifdef WOLFKM_DEBUG_DH
    pr_info("info: exiting km_dh_init: name %d, nbits %d",
            ctx->name, ctx->nbits);
    #endif /* WOLFKM_DEBUG_DH */
    return 0;
}

#ifdef LINUXKM_DH
static int km_dh_init(struct crypto_kpp *tfm)
{
    return km_ffdhe_init(tfm, 0, DH_MAX_SIZE);
}
#endif /* LINUXKM_DH */

#ifdef HAVE_FFDHE_2048
static int km_ffdhe2048_init(struct crypto_kpp *tfm)
{
    return km_ffdhe_init(tfm, WC_FFDHE_2048, 2048);
}
#endif /* HAVE_FFDHE_2048 */

#ifdef HAVE_FFDHE_3072
static int km_ffdhe3072_init(struct crypto_kpp *tfm)
{
    return km_ffdhe_init(tfm, WC_FFDHE_3072, 3072);
}
#endif /* HAVE_FFDHE_3072 */

#ifdef HAVE_FFDHE_4096
static int km_ffdhe4096_init(struct crypto_kpp *tfm)
{
    return km_ffdhe_init(tfm, WC_FFDHE_4096, 4096);
}
#endif /* HAVE_FFDHE_4096 */

#ifdef HAVE_FFDHE_6144
static int km_ffdhe6144_init(struct crypto_kpp *tfm)
{
    return km_ffdhe_init(tfm, WC_FFDHE_6144, 6144);
}
#endif /* HAVE_FFDHE_6144 */

#ifdef HAVE_FFDHE_8192
static int km_ffdhe8192_init(struct crypto_kpp *tfm)
{
    return km_ffdhe_init(tfm, WC_FFDHE_8192, 8192);
}
#endif /* HAVE_FFDHE_8192 */

/**
 * Generate the dh public key:
 *   - req->src should be null
 *   - req->dst is where we place the public key.
 * The kernel api expects raw pub key.
 * */
static int km_dh_gen_pub(struct kpp_request *req)
{
    struct crypto_kpp * tfm = NULL;
    struct km_dh_ctx *  ctx = NULL;
    int                 err = -1;

    if (req->src != NULL || req->dst == NULL) {
        return -EINVAL;
    }

    tfm = crypto_kpp_reqtfm(req);
    ctx = kpp_tfm_ctx(tfm);

    if (!ctx->priv_key || !ctx->pub_key) {
        /* key not set or invalid key state. */
        return -EINVAL;
    }

    if (ctx->has_pub_key == 0) {
        memset(ctx->pub_key, 0, ctx->pub_len);

        err = wc_DhGeneratePublic(ctx->key, ctx->priv_key, ctx->priv_len,
                                  ctx->pub_key, &ctx->pub_len);

        if (unlikely(err)) {
            #ifdef WOLFKM_DEBUG_DH
            pr_err("%s: wc_dh_make_key_ex failed: %d\n",
                   WOLFKM_DH_DRIVER, err);
            #endif
            return -EINVAL;
        }

        ctx->has_pub_key = 1;
    }

    if (ctx->pub_len > req->dst_len) {
        #ifdef WOLFKM_DEBUG_DH
        pr_err("error: dst_len too small: %d", req->dst_len);
        #endif /* WOLFKM_DEBUG_DH */
        req->dst_len = ctx->pub_len;
        return -EOVERFLOW;
    }

    /* copy generated pub to req->dst */
    scatterwalk_map_and_copy(ctx->pub_key, req->dst, 0, ctx->pub_len, 1);
    err = 0;

    #ifdef WOLFKM_DEBUG_DH
    pr_info("info: exiting km_dh_gen_pub: %d", ctx->pub_len);
    #endif /* WOLFKM_DEBUG_DH */
    return err;
}

/**
 * Generate dh shared secret.
 *   - req->src has raw pub key from other party.
 *   - req->dst is shared secret output buffer.
 * */
static int km_dh_compute_shared_secret(struct kpp_request *req)
{
    struct crypto_kpp * tfm = NULL;
    struct km_dh_ctx *  ctx = NULL;
    int                 err = -1;
    byte *              pub = NULL;
    word32              pub_len = 0;
    byte *              shared_secret = NULL;
    word32              shared_secret_len = 0;

    if (req->src == NULL || req->dst == NULL) {
        err = -EINVAL;
        goto dh_shared_secret_end;
    }

    tfm = crypto_kpp_reqtfm(req);
    ctx = kpp_tfm_ctx(tfm);

    if (!ctx->priv_key || !ctx->pub_key) {
        /* key not set or invalid key state. */
        err = -EINVAL;
        goto dh_shared_secret_end;
    }

    if (req->src_len <= 0 || req->src_len > (ctx->nbits / WOLFSSL_BIT_SIZE)) {
        #ifdef WOLFKM_DEBUG_DH
        pr_err("error: got src_len %d, expected %d", req->src_len,
               (ctx->nbits / WOLFSSL_BIT_SIZE));
        #endif /* WOLFKM_DEBUG_DH */
        err = -EINVAL;
        goto dh_shared_secret_end;
    }

    pub_len = req->src_len;

    pub = malloc(pub_len);
    if (!pub) {
        err = -ENOMEM;
        goto dh_shared_secret_end;
    }

    memset(pub, 0, pub_len);

    /* copy req->src to pub */
    scatterwalk_map_and_copy(pub, req->src, 0, req->src_len, 0);

    shared_secret_len = pub_len;
    shared_secret = malloc(shared_secret_len);
    if (!shared_secret) {
        err = -ENOMEM;
        goto dh_shared_secret_end;
    }

    PRIVATE_KEY_UNLOCK();
    err = wc_DhAgree(ctx->key, shared_secret, &shared_secret_len,
                     ctx->priv_key, ctx->priv_len, pub, pub_len);
    PRIVATE_KEY_LOCK();

    if (unlikely(err)) {
        #ifdef WOLFKM_DEBUG_DH
        pr_err("error: wc_dh_shared_secret returned: %d, %d\n", err,
               shared_secret_len);
        #endif
        err = -EINVAL;
        goto dh_shared_secret_end;
    }

    if (req->dst_len < shared_secret_len) {
        #ifdef WOLFKM_DEBUG_DH
        pr_err("error: got dst_len %d, shared_secret_len %d\n",
               req->dst_len, shared_secret_len);
        #endif
        req->dst_len = shared_secret_len;
        err = -EOVERFLOW;
        goto dh_shared_secret_end;
    }

    /* copy shared_secret to req->dst */
    scatterwalk_map_and_copy(shared_secret, req->dst, 0, shared_secret_len, 1);

dh_shared_secret_end:
    if (shared_secret) {
        ForceZero(shared_secret, shared_secret_len);
        free(shared_secret);
        shared_secret = NULL;
    }
    if (pub) { free(pub); pub = NULL; }

    #ifdef WOLFKM_DEBUG_DH
    pr_info("info: exiting km_dh_compute_shared_secret: %d\n",
            shared_secret_len);
    #endif /* WOLFKM_DEBUG_DH */
    return err;
}

#ifdef LINUXKM_DH
static int linuxkm_test_dh(void)
{
    int rc = 0;
    /* reference values from kernel crypto/testmgr.h */
    /* 529 byte secret size */
    static const byte secret[] = {
#ifdef LITTLE_ENDIAN_ORDER
        0x01, 0x00, /* type */
        0x11, 0x02, /* len */
        0x00, 0x01, 0x00, 0x00, /* key_size */
        0x00, 0x01, 0x00, 0x00, /* p_size */
        0x01, 0x00, 0x00, 0x00, /* g_size */
#else
        0x00, 0x01, /* type */
        0x02, 0x11, /* len */
        0x00, 0x00, 0x01, 0x00, /* key_size */
        0x00, 0x00, 0x01, 0x00, /* p_size */
        0x00, 0x00, 0x00, 0x01, /* g_size */
#endif
        /* xa */
        0x44, 0xc1, 0x48, 0x36, 0xa7, 0x2b, 0x6f, 0x4e,
        0x43, 0x03, 0x68, 0xad, 0x31, 0x00, 0xda, 0xf3,
        0x2a, 0x01, 0xa8, 0x32, 0x63, 0x5f, 0x89, 0x32,
        0x1f, 0xdf, 0x4c, 0xa1, 0x6a, 0xbc, 0x10, 0x15,
        0x90, 0x35, 0xc9, 0x26, 0x41, 0xdf, 0x7b, 0xaa,
        0x56, 0x56, 0x3d, 0x85, 0x44, 0xb5, 0xc0, 0x8e,
        0x37, 0x83, 0x06, 0x50, 0xb3, 0x5f, 0x0e, 0x28,
        0x2c, 0xd5, 0x46, 0x15, 0xe3, 0xda, 0x7d, 0x74,
        0x87, 0x13, 0x91, 0x4f, 0xd4, 0x2d, 0xf6, 0xc7,
        0x5e, 0x14, 0x2c, 0x11, 0xc2, 0x26, 0xb4, 0x3a,
        0xe3, 0xb2, 0x36, 0x20, 0x11, 0x3b, 0x22, 0xf2,
        0x06, 0x65, 0x66, 0xe2, 0x57, 0x58, 0xf8, 0x22,
        0x1a, 0x94, 0xbd, 0x2b, 0x0e, 0x8c, 0x55, 0xad,
        0x61, 0x23, 0x45, 0x2b, 0x19, 0x1e, 0x63, 0x3a,
        0x13, 0x61, 0xe3, 0xa0, 0x79, 0x70, 0x3e, 0x6d,
        0x98, 0x32, 0xbc, 0x7f, 0x82, 0xc3, 0x11, 0xd8,
        0xeb, 0x53, 0xb5, 0xfc, 0xb5, 0xd5, 0x3c, 0x4a,
        0xea, 0x92, 0x3e, 0x01, 0xce, 0x15, 0x65, 0xd4,
        0xaa, 0x85, 0xc1, 0x11, 0x90, 0x83, 0x31, 0x6e,
        0xfe, 0xe7, 0x7f, 0x7d, 0xed, 0xab, 0xf9, 0x29,
        0xf8, 0xc7, 0xf1, 0x68, 0xc6, 0xb7, 0xe4, 0x1f,
        0x2f, 0x28, 0xa0, 0xc9, 0x1a, 0x50, 0x64, 0x29,
        0x4b, 0x01, 0x6d, 0x1a, 0xda, 0x46, 0x63, 0x21,
        0x07, 0x40, 0x8c, 0x8e, 0x4c, 0x6f, 0xb5, 0xe5,
        0x12, 0xf3, 0xc2, 0x1b, 0x48, 0x27, 0x5e, 0x27,
        0x01, 0xb1, 0xaa, 0xed, 0x68, 0x9b, 0x83, 0x18,
        0x8f, 0xb1, 0xeb, 0x1f, 0x04, 0xd1, 0x3c, 0x79,
        0xed, 0x4b, 0xf7, 0x0a, 0x33, 0xdc, 0xe0, 0xc6,
        0xd8, 0x02, 0x51, 0x59, 0x00, 0x74, 0x30, 0x07,
        0x4c, 0x2d, 0xac, 0xe4, 0x13, 0xf1, 0x80, 0xf0,
        0xce, 0xfa, 0xff, 0xa9, 0xce, 0x29, 0x46, 0xdd,
        0x9d, 0xad, 0xd1, 0xc3, 0xc6, 0x58, 0x1a, 0x63,
        /* p */
        0xb9, 0x36, 0x3a, 0xf1, 0x82, 0x1f, 0x60, 0xd3,
        0x22, 0x47, 0xb8, 0xbc, 0x2d, 0x22, 0x6b, 0x81,
        0x7f, 0xe8, 0x20, 0x06, 0x09, 0x23, 0x73, 0x49,
        0x9a, 0x59, 0x8b, 0x35, 0x25, 0xf8, 0x31, 0xbc,
        0x7d, 0xa8, 0x1c, 0x9d, 0x56, 0x0d, 0x1a, 0xf7,
        0x4b, 0x4f, 0x96, 0xa4, 0x35, 0x77, 0x6a, 0x89,
        0xab, 0x42, 0x00, 0x49, 0x21, 0x71, 0xed, 0x28,
        0x16, 0x1d, 0x87, 0x5a, 0x10, 0xa7, 0x9c, 0x64,
        0x94, 0xd4, 0x87, 0x3d, 0x28, 0xef, 0x44, 0xfe,
        0x4b, 0xe2, 0xb4, 0x15, 0x8c, 0x82, 0xa6, 0xf3,
        0x50, 0x5f, 0xa8, 0xe8, 0xa2, 0x60, 0xe7, 0x00,
        0x86, 0x78, 0x05, 0xd4, 0x78, 0x19, 0xa1, 0x98,
        0x62, 0x4e, 0x4a, 0x00, 0x78, 0x56, 0x96, 0xe6,
        0xcf, 0xd7, 0x10, 0x1b, 0x74, 0x5d, 0xd0, 0x26,
        0x61, 0xdb, 0x6b, 0x32, 0x09, 0x51, 0xd8, 0xa5,
        0xfd, 0x54, 0x16, 0x71, 0x01, 0xb3, 0x39, 0xe6,
        0x4e, 0x69, 0xb1, 0xd7, 0x06, 0x8f, 0xd6, 0x1e,
        0xdc, 0x72, 0x25, 0x26, 0x74, 0xc8, 0x41, 0x06,
        0x5c, 0xd1, 0x26, 0x5c, 0xb0, 0x2f, 0xf9, 0x59,
        0x13, 0xc1, 0x2a, 0x0f, 0x78, 0xea, 0x7b, 0xf7,
        0xbd, 0x59, 0xa0, 0x90, 0x1d, 0xfc, 0x33, 0x5b,
        0x4c, 0xbf, 0x05, 0x9c, 0x3a, 0x3f, 0x69, 0xa2,
        0x45, 0x61, 0x4e, 0x10, 0x6a, 0xb3, 0x17, 0xc5,
        0x68, 0x30, 0xfb, 0x07, 0x5f, 0x34, 0xc6, 0xfb,
        0x73, 0x07, 0x3c, 0x70, 0xf6, 0xae, 0xe7, 0x72,
        0x84, 0xc3, 0x18, 0x81, 0x8f, 0xe8, 0x11, 0x1f,
        0x3d, 0x83, 0x83, 0x01, 0x2a, 0x14, 0x73, 0xbf,
        0x32, 0x32, 0x2e, 0xc9, 0x4d, 0xdb, 0x2a, 0xca,
        0xee, 0x71, 0xf9, 0xda, 0xad, 0xe8, 0x82, 0x0b,
        0x4d, 0x0c, 0x1f, 0xb6, 0x1d, 0xef, 0x00, 0x67,
        0x74, 0x3d, 0x95, 0xe0, 0xb7, 0xc4, 0x30, 0x8a,
        0x24, 0x87, 0x12, 0x47, 0x27, 0x70, 0x0d, 0x73,
        /* g */
        0x02
    };

    /* 256 byte pub key */
    static const byte b_pub[] = {
        0x2a, 0x67, 0x5c, 0xfd, 0x63, 0x5d, 0xc0, 0x97,
        0x0a, 0x8b, 0xa2, 0x1f, 0xf8, 0x8a, 0xcb, 0x54,
        0xca, 0x2f, 0xd3, 0x49, 0x3f, 0x01, 0x8e, 0x87,
        0xfe, 0xcc, 0x94, 0xa0, 0x3e, 0xd4, 0x26, 0x79,
        0x9a, 0x94, 0x3c, 0x11, 0x81, 0x58, 0x5c, 0x60,
        0x3d, 0xf5, 0x98, 0x90, 0x89, 0x64, 0x62, 0x1f,
        0xbd, 0x05, 0x6d, 0x2b, 0xcd, 0x84, 0x40, 0x9b,
        0x4a, 0x1f, 0xe0, 0x19, 0xf1, 0xca, 0x20, 0xb3,
        0x4e, 0xa0, 0x4f, 0x15, 0xcc, 0xa5, 0xfe, 0xa5,
        0xb4, 0xf5, 0x0b, 0x18, 0x7a, 0x5a, 0x37, 0xaa,
        0x58, 0x00, 0x19, 0x7f, 0xe2, 0xa3, 0xd9, 0x1c,
        0x44, 0x57, 0xcc, 0xde, 0x2e, 0xc1, 0x38, 0xea,
        0xeb, 0xe3, 0x90, 0x40, 0xc4, 0x6c, 0xf7, 0xcd,
        0xe9, 0x22, 0x50, 0x71, 0xf5, 0x7c, 0xdb, 0x37,
        0x0e, 0x80, 0xc3, 0xed, 0x7e, 0xb1, 0x2b, 0x2f,
        0xbe, 0x71, 0xa6, 0x11, 0xa5, 0x9d, 0xf5, 0x39,
        0xf1, 0xa2, 0xe5, 0x85, 0xbc, 0x25, 0x91, 0x4e,
        0x84, 0x8d, 0x26, 0x9f, 0x4f, 0xe6, 0x0f, 0xa6,
        0x2b, 0x6b, 0xf9, 0x0d, 0xaf, 0x6f, 0xbb, 0xfa,
        0x2d, 0x79, 0x15, 0x31, 0x57, 0xae, 0x19, 0x60,
        0x22, 0x0a, 0xf5, 0xfd, 0x98, 0x0e, 0xbf, 0x5d,
        0x49, 0x75, 0x58, 0x37, 0xbc, 0x7f, 0xf5, 0x21,
        0x56, 0x1e, 0xd5, 0xb3, 0x50, 0x0b, 0xca, 0x96,
        0xf3, 0xd1, 0x3f, 0xb3, 0x70, 0xa8, 0x6d, 0x63,
        0x48, 0xfb, 0x3d, 0xd7, 0x29, 0x91, 0x45, 0xb5,
        0x48, 0xcd, 0xb6, 0x78, 0x30, 0xf2, 0x3f, 0x1e,
        0xd6, 0x22, 0xd6, 0x35, 0x9b, 0xf9, 0x1f, 0x85,
        0xae, 0xab, 0x4b, 0xd7, 0xe0, 0xc7, 0x86, 0x67,
        0x3f, 0x05, 0x7f, 0xa6, 0x0d, 0x2f, 0x0d, 0xbf,
        0x53, 0x5f, 0x4d, 0x2c, 0x6d, 0x5e, 0x57, 0x40,
        0x30, 0x3a, 0x23, 0x98, 0xf9, 0xb4, 0x32, 0xf5,
        0x32, 0x83, 0xdd, 0x0b, 0xae, 0x33, 0x97, 0x2f
    };

    /* 256 byte pub key */
    static const byte expected_a_pub[] = {
        0x5c, 0x24, 0xdf, 0xeb, 0x5b, 0x4b, 0xf8, 0xc5,
        0xef, 0x39, 0x48, 0x82, 0xe0, 0x1e, 0x62, 0xee,
        0x8a, 0xae, 0xdf, 0x93, 0x6c, 0x2b, 0x16, 0x95,
        0x92, 0x16, 0x3f, 0x16, 0x7b, 0x75, 0x03, 0x85,
        0xd9, 0xf1, 0x69, 0xc2, 0x14, 0x87, 0x45, 0xfc,
        0xa4, 0x19, 0xf6, 0xf0, 0xa4, 0xf3, 0xec, 0xd4,
        0x6c, 0x5c, 0x03, 0x3b, 0x94, 0xc2, 0x2f, 0x92,
        0xe4, 0xce, 0xb3, 0xe4, 0x72, 0xe8, 0x17, 0xe6,
        0x23, 0x7e, 0x00, 0x01, 0x09, 0x59, 0x13, 0xbf,
        0xc1, 0x2f, 0x99, 0xa9, 0x07, 0xaa, 0x02, 0x23,
        0x4a, 0xca, 0x39, 0x4f, 0xbc, 0xec, 0x0f, 0x27,
        0x4f, 0x19, 0x93, 0x6c, 0xb9, 0x30, 0x52, 0xfd,
        0x2b, 0x9d, 0x86, 0xf1, 0x06, 0x1e, 0xb6, 0x56,
        0x27, 0x4a, 0xc9, 0x8a, 0xa7, 0x8a, 0x48, 0x5e,
        0xb5, 0x60, 0xcb, 0xdf, 0xff, 0x03, 0x26, 0x10,
        0xbf, 0x90, 0x8f, 0x46, 0x60, 0xeb, 0x9b, 0x9a,
        0xd6, 0x6f, 0x44, 0x91, 0x03, 0x92, 0x18, 0x2c,
        0x96, 0x5e, 0x40, 0x19, 0xfb, 0xf4, 0x4f, 0x3a,
        0x02, 0x7b, 0xaf, 0xcc, 0x22, 0x20, 0x79, 0xb9,
        0xf8, 0x9f, 0x8f, 0x85, 0x6b, 0xec, 0x44, 0xbb,
        0xe6, 0xa8, 0x8e, 0xb1, 0xe8, 0x2c, 0xee, 0x64,
        0xee, 0xf8, 0xbd, 0x00, 0xf3, 0xe2, 0x2b, 0x93,
        0xcd, 0xe7, 0xc4, 0xdf, 0xc9, 0x19, 0x46, 0xfe,
        0xb6, 0x07, 0x73, 0xc1, 0x8a, 0x64, 0x79, 0x26,
        0xe7, 0x30, 0xad, 0x2a, 0xdf, 0xe6, 0x8f, 0x59,
        0xf5, 0x81, 0xbf, 0x4a, 0x29, 0x91, 0xe7, 0xb7,
        0xcf, 0x48, 0x13, 0x27, 0x75, 0x79, 0x40, 0xd9,
        0xd6, 0x32, 0x52, 0x4e, 0x6a, 0x86, 0xae, 0x6f,
        0xc2, 0xbf, 0xec, 0x1f, 0xc2, 0x69, 0xb2, 0xb6,
        0x59, 0xe5, 0xa5, 0x17, 0xa4, 0x77, 0xb7, 0x62,
        0x46, 0xde, 0xe8, 0xd2, 0x89, 0x78, 0x9a, 0xef,
        0xa3, 0xb5, 0x8f, 0x26, 0xec, 0x80, 0xda, 0x39
    };

    /* 256 byte shared secret */
    static const byte shared_secret[] = {
        0x8f, 0xf3, 0xac, 0xa2, 0xea, 0x22, 0x11, 0x5c,
        0x45, 0x65, 0x1a, 0x77, 0x75, 0x2e, 0xcf, 0x46,
        0x23, 0x14, 0x1e, 0x67, 0x53, 0x4d, 0x35, 0xb0,
        0x38, 0x1d, 0x4e, 0xb9, 0x41, 0x9a, 0x21, 0x24,
        0x6e, 0x9f, 0x40, 0xfe, 0x90, 0x51, 0xb1, 0x06,
        0xa4, 0x7b, 0x87, 0x17, 0x2f, 0xe7, 0x5e, 0x22,
        0xf0, 0x7b, 0x54, 0x84, 0x0a, 0xac, 0x0a, 0x90,
        0xd2, 0xd7, 0xe8, 0x7f, 0xe7, 0xe3, 0x30, 0x75,
        0x01, 0x1f, 0x24, 0x75, 0x56, 0xbe, 0xcc, 0x8d,
        0x1e, 0x68, 0x0c, 0x41, 0x72, 0xd3, 0xfa, 0xbb,
        0xe5, 0x9c, 0x60, 0xc7, 0x28, 0x77, 0x0c, 0xbe,
        0x89, 0xab, 0x08, 0xd6, 0x21, 0xe7, 0x2e, 0x1a,
        0x58, 0x7a, 0xca, 0x4f, 0x22, 0xf3, 0x2b, 0x30,
        0xfd, 0xf4, 0x98, 0xc1, 0xa3, 0xf8, 0xf6, 0xcc,
        0xa9, 0xe4, 0xdb, 0x5b, 0xee, 0xd5, 0x5c, 0x6f,
        0x62, 0x4c, 0xd1, 0x1a, 0x02, 0x2a, 0x23, 0xe4,
        0xb5, 0x57, 0xf3, 0xf9, 0xec, 0x04, 0x83, 0x54,
        0xfe, 0x08, 0x5e, 0x35, 0xac, 0xfb, 0xa8, 0x09,
        0x82, 0x32, 0x60, 0x11, 0xb2, 0x16, 0x62, 0x6b,
        0xdf, 0xda, 0xde, 0x9c, 0xcb, 0x63, 0x44, 0x6c,
        0x59, 0x26, 0x6a, 0x8f, 0xb0, 0x24, 0xcb, 0xa6,
        0x72, 0x48, 0x1e, 0xeb, 0xe0, 0xe1, 0x09, 0x44,
        0xdd, 0xee, 0x66, 0x6d, 0x84, 0xcf, 0xa5, 0xc1,
        0xb8, 0x36, 0x74, 0xd3, 0x15, 0x96, 0xc3, 0xe4,
        0xc6, 0x5a, 0x4d, 0x23, 0x97, 0x0c, 0x5c, 0xcb,
        0xa9, 0xf5, 0x29, 0xc2, 0x0e, 0xff, 0x93, 0x82,
        0xd3, 0x34, 0x49, 0xad, 0x64, 0xa6, 0xb1, 0xc0,
        0x59, 0x28, 0x75, 0x60, 0xa7, 0x8a, 0xb0, 0x11,
        0x56, 0x89, 0x42, 0x74, 0x11, 0xf5, 0xf6, 0x5e,
        0x6f, 0x16, 0x54, 0x6a, 0xb1, 0x76, 0x4d, 0x50,
        0x8a, 0x68, 0xc1, 0x5b, 0x82, 0xb9, 0x0d, 0x00,
        0x32, 0x50, 0xed, 0x88, 0x87, 0x48, 0x92, 0x17
    };

    rc = linuxkm_test_kpp_driver(WOLFKM_DH_DRIVER,
                                 b_pub, expected_a_pub, sizeof(b_pub),
                                 secret, sizeof(secret),
                                 shared_secret, sizeof(shared_secret));

    return rc;
}
#endif /* LINUXKM_DH */

#ifdef HAVE_FFDHE_2048
static int linuxkm_test_ffdhe2048(void)
{
    int rc = 0;

    /* reference values from kernel crypto/testmgr.h */
    /* 272 byte secret size */
    static const byte secret[] = {
#ifdef LITTLE_ENDIAN_ORDER
        0x01, 0x00, /* type */
        0x10, 0x01, /* len */
        0x00, 0x01, 0x00, 0x00, /* key_size */
        0x00, 0x00, 0x00, 0x00, /* p_size */
        0x00, 0x00, 0x00, 0x00, /* g_size */
#else
        0x00, 0x01, /* type */
        0x01, 0x10, /* len */
        0x00, 0x00, 0x01, 0x00, /* key_size */
        0x00, 0x00, 0x00, 0x00, /* p_size */
        0x00, 0x00, 0x00, 0x00, /* g_size */
#endif
        /* xa */
        0x23, 0x7d, 0xd0, 0x06, 0xfd, 0x7a, 0xe5, 0x7a,
        0x08, 0xda, 0x98, 0x31, 0xc0, 0xb3, 0xd5, 0x85,
        0xe2, 0x0d, 0x2a, 0x91, 0x5f, 0x78, 0x4b, 0xa6,
        0x62, 0xd0, 0xa6, 0x35, 0xd4, 0xef, 0x86, 0x39,
        0xf1, 0xdb, 0x71, 0x5e, 0xb0, 0x11, 0x2e, 0xee,
        0x91, 0x3a, 0xaa, 0xf9, 0xe3, 0xdf, 0x8d, 0x8b,
        0x48, 0x41, 0xde, 0xe8, 0x78, 0x53, 0xc5, 0x5f,
        0x93, 0xd2, 0x79, 0x0d, 0xbe, 0x8d, 0x83, 0xe8,
        0x8f, 0x00, 0xd2, 0xde, 0x13, 0x18, 0x04, 0x05,
        0x20, 0x6d, 0xda, 0xfa, 0x1d, 0x0b, 0x24, 0x52,
        0x3a, 0x18, 0x2b, 0xe1, 0x1e, 0xae, 0x15, 0x3b,
        0x0f, 0xaa, 0x09, 0x09, 0xf6, 0x01, 0x98, 0xe9,
        0x81, 0x5d, 0x6b, 0x83, 0x6e, 0x55, 0xf1, 0x5d,
        0x6f, 0x6f, 0x0d, 0x9d, 0xa8, 0x72, 0x32, 0x63,
        0x60, 0xe6, 0x0b, 0xc5, 0x22, 0xe2, 0xf9, 0x46,
        0x58, 0xa2, 0x1c, 0x2a, 0xb0, 0xd5, 0xaf, 0xe3,
        0x5b, 0x03, 0xb7, 0x36, 0xb7, 0xba, 0x55, 0x20,
        0x08, 0x7c, 0x51, 0xd4, 0x89, 0x42, 0x9c, 0x14,
        0x23, 0xe2, 0x71, 0x3e, 0x15, 0x2a, 0x0d, 0x34,
        0x8a, 0xde, 0xad, 0x84, 0x11, 0x15, 0x72, 0x18,
        0x42, 0x43, 0x0a, 0xe2, 0x58, 0x29, 0xb3, 0x90,
        0x0f, 0x56, 0xd8, 0x8a, 0x0f, 0x0e, 0xbc, 0x0e,
        0x9c, 0xe7, 0xd5, 0xe6, 0x5b, 0xbf, 0x06, 0x64,
        0x38, 0x12, 0xa5, 0x8d, 0x5b, 0x68, 0x34, 0xdd,
        0x75, 0x48, 0xc9, 0xa7, 0xa3, 0x58, 0x5a, 0x1c,
        0xe1, 0xb2, 0xc5, 0xe3, 0x39, 0x03, 0xcf, 0xab,
        0xc2, 0x14, 0x07, 0xaf, 0x55, 0x80, 0xc7, 0x63,
        0xe4, 0x03, 0xeb, 0xe9, 0x0a, 0x25, 0x61, 0x85,
        0x1d, 0x0e, 0x81, 0x52, 0x7b, 0xbc, 0x4a, 0x0c,
        0xc8, 0x59, 0x6a, 0xac, 0x18, 0xfb, 0x8c, 0x0c,
        0xb4, 0x79, 0xbd, 0xa1, 0x4c, 0xbb, 0x02, 0xc9,
        0xd5, 0x13, 0x88, 0x3d, 0x25, 0xaa, 0x77, 0x49
    };

    /* 256 byte pub key */
    static const byte b_pub[] = {
        0x5c, 0x00, 0x6f, 0xda, 0xfe, 0x4c, 0x0c, 0xc2,
        0x18, 0xff, 0xa9, 0xec, 0x7a, 0xbe, 0x8a, 0x51,
        0x64, 0x6b, 0x57, 0xf8, 0xed, 0xe2, 0x36, 0x77,
        0xc1, 0x23, 0xbf, 0x56, 0xa6, 0x48, 0x76, 0x34,
        0x0e, 0xf3, 0x68, 0x05, 0x45, 0x6a, 0x98, 0x5b,
        0x9e, 0x8b, 0xc0, 0x11, 0x29, 0xcb, 0x5b, 0x66,
        0x2d, 0xc2, 0xeb, 0x4c, 0xf1, 0x7d, 0x85, 0x30,
        0xaa, 0xd5, 0xf5, 0xb8, 0xd3, 0x62, 0x1e, 0x97,
        0x1e, 0x34, 0x18, 0xf8, 0x76, 0x8c, 0x10, 0xca,
        0x1f, 0xe4, 0x5d, 0x62, 0xe1, 0xbe, 0x61, 0xef,
        0xaf, 0x2c, 0x8d, 0x97, 0x15, 0xa5, 0x86, 0xd5,
        0xd3, 0x12, 0x6f, 0xec, 0xe2, 0xa4, 0xb2, 0x5a,
        0x35, 0x1d, 0xd4, 0x91, 0xa6, 0xef, 0x13, 0x09,
        0x65, 0x9c, 0x45, 0xc0, 0x12, 0xad, 0x7f, 0xee,
        0x93, 0x5d, 0xfa, 0x89, 0x26, 0x7d, 0xae, 0xee,
        0xea, 0x8c, 0xa3, 0xcf, 0x04, 0x2d, 0xa0, 0xc7,
        0xd9, 0x14, 0x62, 0xaf, 0xdf, 0xa0, 0x33, 0xd7,
        0x5e, 0x83, 0xa2, 0xe6, 0x0e, 0x0e, 0x5d, 0x77,
        0xce, 0xe6, 0x72, 0xe4, 0xec, 0x9d, 0xff, 0x72,
        0x9f, 0x38, 0x95, 0x19, 0x96, 0xba, 0x4c, 0xe3,
        0x5f, 0xb8, 0x46, 0x4a, 0x1d, 0xe9, 0x62, 0x7b,
        0xa8, 0xdc, 0xe7, 0x61, 0x90, 0x6b, 0xb9, 0xd4,
        0xad, 0x0b, 0xa3, 0x06, 0xb3, 0x70, 0xfa, 0xea,
        0x2b, 0xc4, 0x2c, 0xde, 0x43, 0x37, 0xf6, 0x8d,
        0x72, 0xf0, 0x86, 0x9a, 0xbb, 0x3b, 0x8e, 0x7a,
        0x71, 0x03, 0x30, 0x30, 0x2a, 0x5d, 0xcd, 0x1e,
        0xe4, 0xd3, 0x08, 0x07, 0x75, 0x17, 0x17, 0x72,
        0x1e, 0x77, 0x6c, 0x98, 0x0d, 0x29, 0x7f, 0xac,
        0xe7, 0xb2, 0xee, 0xa9, 0x1c, 0x33, 0x9d, 0x08,
        0x39, 0xe1, 0xd8, 0x5b, 0xe5, 0xbc, 0x48, 0xb2,
        0xb6, 0xdf, 0xcd, 0xa0, 0x42, 0x06, 0xcc, 0xfb,
        0xed, 0x60, 0x6f, 0xbc, 0x57, 0xac, 0x09, 0x45
    };

    /* 256 byte pub key */
    static const byte expected_a_pub[] = {
        0x8b, 0xdb, 0xc1, 0xf7, 0xc6, 0xba, 0xa1, 0x38,
        0x95, 0x6a, 0xa1, 0xb6, 0x04, 0x5e, 0xae, 0x52,
        0x72, 0xfc, 0xef, 0x2d, 0x9d, 0x71, 0x05, 0x9c,
        0xd3, 0x02, 0xa9, 0xfb, 0x55, 0x0f, 0xfa, 0xc9,
        0xb4, 0x34, 0x51, 0xa3, 0x28, 0x89, 0x8d, 0x93,
        0x92, 0xcb, 0xd9, 0xb5, 0xb9, 0x66, 0xfc, 0x67,
        0x15, 0x92, 0x6f, 0x73, 0x85, 0x15, 0xe2, 0xfc,
        0x11, 0x6b, 0x97, 0x8c, 0x4b, 0x0f, 0x12, 0xfa,
        0x8d, 0x72, 0x76, 0x9b, 0x8f, 0x3b, 0xfe, 0x31,
        0xbe, 0x42, 0x88, 0x4c, 0xd2, 0xb2, 0x70, 0xa6,
        0xa5, 0xe3, 0x7e, 0x73, 0x07, 0x12, 0x36, 0xaa,
        0xc9, 0x5c, 0x83, 0xe1, 0xf1, 0x46, 0x41, 0x4f,
        0x7c, 0x52, 0xaf, 0xdc, 0xa4, 0xe6, 0x82, 0xa3,
        0x86, 0x83, 0x47, 0x5a, 0x12, 0x3a, 0x0c, 0xe3,
        0xdd, 0xdb, 0x94, 0x03, 0x2a, 0x59, 0x91, 0xa0,
        0x19, 0xe5, 0xf8, 0x07, 0xdd, 0x54, 0x6a, 0x22,
        0x43, 0xb7, 0xf3, 0x74, 0xd7, 0xb9, 0x30, 0xfe,
        0x9c, 0xe8, 0xd1, 0xcf, 0x06, 0x43, 0x68, 0xb9,
        0x54, 0x8f, 0x54, 0xa2, 0xe5, 0x3c, 0xf2, 0xc3,
        0x4c, 0xee, 0xd4, 0x7c, 0x5d, 0x0e, 0xb1, 0x7b,
        0x16, 0x68, 0xb5, 0xb3, 0x7d, 0xd4, 0x11, 0x83,
        0x5c, 0x77, 0x17, 0xc4, 0xf0, 0x59, 0x76, 0x7a,
        0x83, 0x40, 0xe5, 0xd9, 0x4c, 0x76, 0x23, 0x5b,
        0x17, 0x6d, 0xee, 0x4a, 0x92, 0x68, 0x4b, 0x89,
        0xa0, 0x6d, 0x23, 0x8c, 0x80, 0x31, 0x33, 0x3a,
        0x12, 0xf4, 0x50, 0xa6, 0xcb, 0x13, 0x97, 0x01,
        0xb8, 0x2c, 0xe6, 0xd2, 0x38, 0xdf, 0xd0, 0x7f,
        0xc6, 0x27, 0x19, 0x0e, 0xb2, 0x07, 0xfd, 0x1f,
        0x1b, 0x9c, 0x1b, 0x87, 0xf9, 0x73, 0x6a, 0x3f,
        0x7f, 0xb0, 0xf9, 0x2f, 0x3c, 0x19, 0x9f, 0xc9,
        0x8f, 0x97, 0x21, 0x0e, 0x8e, 0xbb, 0x1a, 0x17,
        0x20, 0x15, 0xdd, 0xc6, 0x42, 0x60, 0xae, 0x4d
    };

    /* 256 byte shared secret */
    static const byte shared_secret[] = {
        0xf3, 0x0e, 0x64, 0x7b, 0x66, 0xd7, 0x82, 0x7e,
        0xab, 0x7e, 0x4a, 0xbe, 0x13, 0x6f, 0x43, 0x3d,
        0xea, 0x4f, 0x1f, 0x8b, 0x9d, 0x41, 0x56, 0x71,
        0xe1, 0x06, 0x96, 0x02, 0x68, 0xfa, 0x44, 0x6e,
        0xe7, 0xf2, 0x26, 0xd4, 0x01, 0x4a, 0xf0, 0x28,
        0x25, 0x76, 0xad, 0xd7, 0xe0, 0x17, 0x74, 0xfe,
        0xf9, 0xe1, 0x6d, 0xd3, 0xf7, 0xc7, 0xdf, 0xc0,
        0x62, 0xa5, 0xf3, 0x4e, 0x1b, 0x5c, 0x77, 0x2a,
        0xfb, 0x0b, 0x87, 0xc3, 0xde, 0x1e, 0xc1, 0xe0,
        0xd3, 0x7a, 0xb8, 0x02, 0x02, 0xec, 0x9c, 0x97,
        0xfb, 0x34, 0xa0, 0x20, 0x10, 0x23, 0x87, 0xb2,
        0x9a, 0x72, 0xe3, 0x3d, 0xb2, 0x18, 0x50, 0xf3,
        0x6a, 0xd3, 0xd3, 0x19, 0xc4, 0x36, 0xd5, 0x59,
        0xd6, 0xd6, 0xa7, 0x5c, 0xc3, 0xf9, 0x09, 0x33,
        0xa1, 0xf5, 0xb9, 0x4b, 0xf3, 0x0b, 0xe1, 0x4f,
        0x79, 0x6b, 0x45, 0xf2, 0xec, 0x8b, 0xe5, 0x69,
        0x9f, 0xc6, 0x05, 0x01, 0xfe, 0x3a, 0x13, 0xfd,
        0x6d, 0xea, 0x03, 0x83, 0x29, 0x7c, 0x7f, 0xf5,
        0x41, 0x55, 0x95, 0xde, 0x7e, 0x62, 0xae, 0xaf,
        0x28, 0xdb, 0x7c, 0xa9, 0x90, 0x1e, 0xb2, 0xb1,
        0x1b, 0xef, 0xf1, 0x2e, 0xde, 0x47, 0xaa, 0xa8,
        0x92, 0x9a, 0x49, 0x3d, 0xc0, 0xe0, 0x8d, 0xbb,
        0x0c, 0x42, 0x86, 0xaf, 0x00, 0xce, 0xb0, 0xab,
        0x22, 0x7c, 0xe9, 0xbe, 0xb9, 0x72, 0x2f, 0xcf,
        0x5e, 0x5d, 0x62, 0x52, 0x2a, 0xd1, 0xfe, 0xcc,
        0xa2, 0xf3, 0x40, 0xfd, 0x01, 0xa7, 0x54, 0x0a,
        0xa1, 0xfb, 0x1c, 0xf2, 0x44, 0xa6, 0x47, 0x30,
        0x5a, 0xba, 0x2a, 0x05, 0xff, 0xd0, 0x6c, 0xab,
        0xeb, 0xe6, 0x8f, 0xf6, 0xd7, 0x73, 0xa3, 0x0e,
        0x6c, 0x0e, 0xcf, 0xfd, 0x8e, 0x16, 0x5d, 0xe0,
        0x2c, 0x11, 0x05, 0x82, 0x3c, 0x22, 0x16, 0x6c,
        0x52, 0x61, 0xcf, 0xbb, 0xff, 0xf8, 0x06, 0xd0
    };

    rc = linuxkm_test_kpp_driver(WOLFKM_FFDHE2048_DRIVER,
                                 b_pub, expected_a_pub, sizeof(b_pub),
                                 secret, sizeof(secret),
                                 shared_secret, sizeof(shared_secret));

    return rc;
}
#endif /* HAVE_FFDHE_2048 */

#ifdef HAVE_FFDHE_3072
static int linuxkm_test_ffdhe3072(void)
{
    int rc = 0;

    /* reference values from kernel crypto/testmgr.h */
    /* 400 byte secret size */
    static const byte secret[] = {
#ifdef LITTLE_ENDIAN_ORDER
        0x01, 0x00, /* type */
        0x90, 0x01, /* len */
        0x80, 0x01, 0x00, 0x00, /* key_size */
        0x00, 0x00, 0x00, 0x00, /* p_size */
        0x00, 0x00, 0x00, 0x00, /* g_size */
#else
        0x00, 0x01, /* type */
        0x01, 0x90, /* len */
        0x00, 0x00, 0x01, 0x80, /* key_size */
        0x00, 0x00, 0x00, 0x00, /* p_size */
        0x00, 0x00, 0x00, 0x00, /* g_size */
#endif
        /* xa */
        0x6b, 0xb4, 0x97, 0x23, 0xfa, 0xc8, 0x5e, 0xa9,
        0x7b, 0x63, 0xe7, 0x3e, 0x0e, 0x99, 0xc3, 0xb9,
        0xda, 0xb7, 0x48, 0x0d, 0xc3, 0xb1, 0xbf, 0x4f,
        0x17, 0xc7, 0xa9, 0x51, 0xf6, 0x64, 0xff, 0xc4,
        0x31, 0x58, 0x87, 0x25, 0x83, 0x2c, 0x00, 0xf0,
        0x41, 0x29, 0xf7, 0xee, 0xf9, 0xe6, 0x36, 0x76,
        0xd6, 0x3a, 0x24, 0xbe, 0xa7, 0x07, 0x0b, 0x93,
        0xc7, 0x9f, 0x6c, 0x75, 0x0a, 0x26, 0x75, 0x76,
        0xe3, 0x0c, 0x42, 0xe0, 0x00, 0x04, 0x69, 0xd9,
        0xec, 0x0b, 0x59, 0x54, 0x28, 0x8f, 0xd7, 0x9a,
        0x63, 0xf4, 0x5b, 0xdf, 0x85, 0x65, 0xc4, 0xe1,
        0x95, 0x27, 0x4a, 0x42, 0xad, 0x36, 0x47, 0xa9,
        0x0a, 0xf8, 0x14, 0x1c, 0xf3, 0x94, 0x3b, 0x7e,
        0x47, 0x99, 0x35, 0xa8, 0x18, 0xec, 0x70, 0x10,
        0xdf, 0xcb, 0xd2, 0x78, 0x88, 0xc1, 0x2d, 0x59,
        0x93, 0xc1, 0xa4, 0x6d, 0xd7, 0x1d, 0xb9, 0xd5,
        0xf8, 0x30, 0x06, 0x7f, 0x98, 0x90, 0x0c, 0x74,
        0x5e, 0x89, 0x2f, 0x64, 0x5a, 0xad, 0x5f, 0x53,
        0xb2, 0xa3, 0xa8, 0x83, 0xbf, 0xfc, 0x37, 0xef,
        0xb8, 0x36, 0x0a, 0x5c, 0x62, 0x81, 0x64, 0x74,
        0x16, 0x2f, 0x45, 0x39, 0x2a, 0x91, 0x26, 0x87,
        0xc0, 0x12, 0xcc, 0x75, 0x11, 0xa3, 0xa1, 0xc5,
        0xae, 0x20, 0xcf, 0xcb, 0x20, 0x25, 0x6b, 0x7a,
        0x31, 0x93, 0x9d, 0x38, 0xb9, 0x57, 0x72, 0x46,
        0xd4, 0x84, 0x65, 0x87, 0xf1, 0xb5, 0xd3, 0xab,
        0xfc, 0xc3, 0x4d, 0x40, 0x92, 0x94, 0x1e, 0xcd,
        0x1c, 0x87, 0xec, 0x3f, 0xcd, 0xbe, 0xd0, 0x95,
        0x6b, 0x40, 0x02, 0xdd, 0x62, 0xeb, 0x0a, 0xda,
        0x4f, 0xbe, 0x8e, 0x32, 0x48, 0x8b, 0x6d, 0x83,
        0xa0, 0x96, 0x62, 0x23, 0xec, 0x83, 0x91, 0x44,
        0xf9, 0x72, 0x01, 0xac, 0xa0, 0xe4, 0x72, 0x1d,
        0x5a, 0x75, 0x05, 0x57, 0x90, 0xae, 0x7e, 0xb4,
        0x71, 0x39, 0x01, 0x05, 0xdc, 0xe9, 0xee, 0xcb,
        0xf0, 0x61, 0x28, 0x91, 0x69, 0x8c, 0x31, 0x03,
        0x7a, 0x92, 0x15, 0xa1, 0x58, 0x67, 0x3d, 0x70,
        0x82, 0xa6, 0x2c, 0xfe, 0x10, 0x56, 0x58, 0xd3,
        0x94, 0x67, 0xe1, 0xbe, 0xee, 0xc1, 0x64, 0x5c,
        0x4b, 0xc8, 0x28, 0x3d, 0xc5, 0x66, 0x3a, 0xab,
        0x22, 0xc1, 0x7e, 0xa1, 0xbb, 0xf3, 0x19, 0x3b,
        0xda, 0x46, 0x82, 0x45, 0xd4, 0x3c, 0x7c, 0xc6,
        0xce, 0x1f, 0x7f, 0x95, 0xa2, 0x17, 0xff, 0x88,
        0xba, 0xd6, 0x4d, 0xdb, 0xd2, 0xea, 0xde, 0x39,
        0xd6, 0xa5, 0x18, 0x73, 0xbb, 0x64, 0x6e, 0x79,
        0xe9, 0xdc, 0x3f, 0x92, 0x7f, 0xda, 0x1f, 0x49,
        0x33, 0x70, 0x65, 0x73, 0xa2, 0xd9, 0x06, 0xb8,
        0x1b, 0x29, 0x29, 0x1a, 0xe0, 0xa3, 0xe6, 0x05,
        0x9a, 0xa8, 0xc2, 0x4e, 0x7a, 0x78, 0x1d, 0x22,
        0x57, 0x21, 0xc8, 0xa3, 0x8d, 0x66, 0x3e, 0x23
    };

    /* 384 byte pub key */
    static const byte b_pub[] = {
        0x73, 0x40, 0x8b, 0xce, 0xe8, 0x6a, 0x1c, 0x03,
        0x50, 0x54, 0x42, 0x36, 0x22, 0xc6, 0x1d, 0xe8,
        0xe1, 0xef, 0x5c, 0x89, 0xa5, 0x55, 0xc1, 0xc4,
        0x1c, 0xd7, 0x4f, 0xee, 0x5d, 0xba, 0x62, 0x60,
        0xfe, 0x93, 0x2f, 0xfd, 0x93, 0x2c, 0x8f, 0x70,
        0xc6, 0x47, 0x17, 0x25, 0xb2, 0x95, 0xd7, 0x7d,
        0x41, 0x81, 0x4d, 0x52, 0x1c, 0xbe, 0x4d, 0x57,
        0x3e, 0x26, 0x51, 0x28, 0x03, 0x8f, 0x67, 0xf5,
        0x22, 0x16, 0x1c, 0x67, 0xf7, 0x62, 0xcb, 0xfd,
        0xa3, 0xee, 0x8d, 0xe0, 0xfa, 0x15, 0x9a, 0x53,
        0xbe, 0x7b, 0x9f, 0xc0, 0x12, 0x7a, 0xfc, 0x5e,
        0x77, 0x2d, 0x60, 0x06, 0xba, 0x71, 0xc5, 0xca,
        0xd7, 0x26, 0xaf, 0x3b, 0xba, 0x6f, 0xd3, 0xc4,
        0x82, 0x57, 0x19, 0x26, 0xb0, 0x16, 0x7b, 0xbd,
        0x83, 0xf2, 0x21, 0x03, 0x79, 0xff, 0x0a, 0x6f,
        0xc5, 0x7b, 0x00, 0x15, 0xad, 0x5b, 0xf4, 0x42,
        0x1f, 0xcb, 0x7f, 0x3d, 0x34, 0x77, 0x3c, 0xc3,
        0xe0, 0x38, 0xa5, 0x40, 0x51, 0xbe, 0x6f, 0xd9,
        0xc9, 0x77, 0x9c, 0xfc, 0x0d, 0xc1, 0x8e, 0xef,
        0x0f, 0xaa, 0x5e, 0xa8, 0xbb, 0x16, 0x4a, 0x3e,
        0x26, 0x55, 0xae, 0xc1, 0xb6, 0x3e, 0xfd, 0x73,
        0xf7, 0x59, 0xd2, 0xe5, 0x4b, 0x91, 0x8e, 0x28,
        0x77, 0x1e, 0x5a, 0xe2, 0xcd, 0xce, 0x92, 0x35,
        0xbb, 0x1e, 0xbb, 0xcf, 0x79, 0x94, 0xdf, 0x31,
        0xde, 0x31, 0xa8, 0x75, 0xf6, 0xe0, 0xaa, 0x2e,
        0xe9, 0x4f, 0x44, 0xc8, 0xba, 0xb9, 0xab, 0x80,
        0x29, 0xa1, 0xea, 0x58, 0x2e, 0x40, 0x96, 0xa0,
        0x1a, 0xf5, 0x2c, 0x38, 0x47, 0x43, 0x5d, 0x26,
        0x2c, 0xd8, 0xad, 0xea, 0xd3, 0xad, 0xe8, 0x51,
        0x49, 0xad, 0x45, 0x2b, 0x25, 0x7c, 0xde, 0xe4,
        0xaf, 0x03, 0x2a, 0x39, 0x26, 0x86, 0x66, 0x10,
        0xbc, 0xa8, 0x71, 0xda, 0xe0, 0xe8, 0xf1, 0xdd,
        0x50, 0xff, 0x44, 0xb2, 0xd3, 0xc7, 0xff, 0x66,
        0x63, 0xf6, 0x42, 0xe3, 0x97, 0x9d, 0x9e, 0xf4,
        0xa6, 0x89, 0xb9, 0xab, 0x12, 0x17, 0xf2, 0x85,
        0x56, 0x9c, 0x6b, 0x24, 0x71, 0x83, 0x57, 0x7d,
        0x3c, 0x7b, 0x2b, 0x88, 0x92, 0x19, 0xd7, 0x1a,
        0x00, 0xd5, 0x38, 0x94, 0x43, 0x60, 0x4d, 0xa7,
        0x12, 0x9e, 0x0d, 0xf6, 0x5c, 0x9a, 0xd3, 0xe2,
        0x9e, 0xb1, 0x21, 0xe8, 0xe2, 0x9e, 0xe9, 0x1e,
        0x9d, 0xa5, 0x94, 0x95, 0xa6, 0x3d, 0x12, 0x15,
        0xd8, 0x8b, 0xac, 0xe0, 0x8c, 0xde, 0xe6, 0x40,
        0x98, 0xaa, 0x5e, 0x55, 0x4f, 0x3d, 0x86, 0x87,
        0x0d, 0xe3, 0xc6, 0x68, 0x15, 0xe6, 0xde, 0x17,
        0x78, 0x21, 0xc8, 0x6c, 0x06, 0xc7, 0x94, 0x56,
        0xb4, 0xaf, 0xa2, 0x35, 0x0b, 0x0c, 0x97, 0xd7,
        0xa4, 0x12, 0xee, 0xf4, 0xd2, 0xef, 0x80, 0x28,
        0xb3, 0xee, 0xe9, 0x15, 0x8b, 0x01, 0x32, 0x79
    };

    /* 384 byte pub key */
    static const byte expected_a_pub[] = {
        0x1b, 0x6a, 0xba, 0xea, 0xa3, 0xcc, 0x50, 0x69,
        0xa9, 0x41, 0x89, 0xaf, 0x04, 0xe1, 0x44, 0x22,
        0x97, 0x20, 0xd1, 0xf6, 0x1e, 0xcb, 0x64, 0x36,
        0x6f, 0xee, 0x0b, 0x16, 0xc1, 0xd9, 0x91, 0xbe,
        0x57, 0xc8, 0xd9, 0xf2, 0xa1, 0x96, 0x91, 0xec,
        0x41, 0xc7, 0x79, 0x00, 0x1a, 0x48, 0x25, 0x55,
        0xbe, 0xf3, 0x20, 0x8c, 0x38, 0xc6, 0x7b, 0xf2,
        0x8b, 0x5a, 0xc3, 0xb5, 0x87, 0x0a, 0x86, 0x3d,
        0xb7, 0xd6, 0xce, 0xb0, 0x96, 0x2e, 0x5d, 0xc4,
        0x00, 0x5e, 0x42, 0xe4, 0xe5, 0x50, 0x4f, 0xb8,
        0x6f, 0x18, 0xa4, 0xe1, 0xd3, 0x20, 0xfc, 0x3c,
        0xf5, 0x0a, 0xff, 0x23, 0xa6, 0x5b, 0xb4, 0x17,
        0x3e, 0x7b, 0xdf, 0xb9, 0xb5, 0x3c, 0x1b, 0x76,
        0x29, 0xcd, 0xb4, 0x46, 0x4f, 0x27, 0x8f, 0xd2,
        0xe8, 0x27, 0x66, 0xdb, 0xe8, 0xb3, 0xf5, 0xe1,
        0xd0, 0x04, 0xcd, 0x89, 0xff, 0xba, 0x76, 0x67,
        0xe8, 0x4d, 0xcf, 0x86, 0x1c, 0x8a, 0xd1, 0xcf,
        0x99, 0x27, 0xfb, 0xa9, 0x78, 0xcc, 0x94, 0xaf,
        0x3d, 0x04, 0xfd, 0x25, 0xc0, 0x47, 0xfa, 0x29,
        0x80, 0x05, 0xf4, 0xde, 0xad, 0xdb, 0xab, 0x12,
        0xb0, 0x2b, 0x8e, 0xca, 0x02, 0x06, 0x6d, 0xad,
        0x3e, 0x09, 0xb1, 0x22, 0xa3, 0xf5, 0x4c, 0x6d,
        0x69, 0x99, 0x58, 0x8b, 0xd8, 0x45, 0x2e, 0xe0,
        0xc9, 0x3c, 0xf7, 0x92, 0xce, 0x21, 0x90, 0x6b,
        0x3b, 0x65, 0x9f, 0x64, 0x79, 0x8d, 0x67, 0x22,
        0x1a, 0x37, 0xd3, 0xee, 0x51, 0xe2, 0xe7, 0x5a,
        0x93, 0x51, 0xaa, 0x3c, 0x4b, 0x04, 0x16, 0x32,
        0xef, 0xe3, 0x66, 0xbe, 0x18, 0x94, 0x88, 0x64,
        0x79, 0xce, 0x06, 0x3f, 0xb8, 0xd6, 0xee, 0xdc,
        0x13, 0x79, 0x6f, 0x20, 0x14, 0xc2, 0x6b, 0xce,
        0xc8, 0xda, 0x42, 0xa5, 0x93, 0x5b, 0xe4, 0x7f,
        0x1a, 0xe6, 0xda, 0x0f, 0xb3, 0xc1, 0x5f, 0x30,
        0x50, 0x76, 0xe8, 0x37, 0x3d, 0xca, 0x77, 0x2c,
        0xa8, 0xe4, 0x3b, 0xf9, 0x6f, 0xe0, 0x17, 0xed,
        0x0e, 0xef, 0xb7, 0x31, 0x14, 0xb5, 0xea, 0xd9,
        0x39, 0x22, 0x89, 0xb6, 0x40, 0x57, 0xcc, 0x84,
        0xef, 0x73, 0xa7, 0xe9, 0x27, 0x21, 0x85, 0x89,
        0xfa, 0xaf, 0x03, 0xda, 0x9c, 0x8b, 0xfd, 0x52,
        0x7d, 0xb0, 0xa4, 0xe4, 0xf9, 0xd8, 0x90, 0x55,
        0xc4, 0x39, 0xd6, 0x9d, 0xaf, 0x3b, 0xce, 0xac,
        0xaa, 0x36, 0x14, 0x7a, 0x9b, 0x8b, 0x12, 0x43,
        0xe1, 0xca, 0x61, 0xae, 0x46, 0x5b, 0xe7, 0xe5,
        0x88, 0x32, 0x80, 0xa0, 0x2d, 0x51, 0xbb, 0x2f,
        0xea, 0xeb, 0x3c, 0x71, 0xb2, 0xae, 0xce, 0xca,
        0x61, 0xd2, 0x76, 0xe0, 0x45, 0x46, 0x78, 0x4e,
        0x09, 0x2d, 0xc2, 0x54, 0xc2, 0xa9, 0xc7, 0xa8,
        0x55, 0x8e, 0x72, 0xa4, 0x8b, 0x8a, 0xc9, 0x01,
        0xdb, 0xe9, 0x58, 0x11, 0xa1, 0xc4, 0xe7, 0x12
    };

    /* 384 byte shared secret */
    static const byte shared_secret[] = {
        0x47, 0x8e, 0xb2, 0x19, 0x09, 0xf0, 0x46, 0x99,
        0x6b, 0x41, 0x86, 0xf7, 0x34, 0xad, 0xbf, 0x2a,
        0x18, 0x1b, 0x7d, 0xec, 0xa9, 0xb2, 0x47, 0x2f,
        0x40, 0xfb, 0x9a, 0x64, 0x30, 0x44, 0xf3, 0x4c,
        0x01, 0x67, 0xad, 0x57, 0x5a, 0xbc, 0xd4, 0xc8,
        0xef, 0x7e, 0x8a, 0x14, 0x74, 0x1d, 0x6d, 0x8c,
        0x7b, 0xce, 0xc5, 0x57, 0x5f, 0x95, 0xe8, 0x72,
        0xba, 0xdf, 0xa3, 0xcd, 0x00, 0xbe, 0x09, 0x4c,
        0x06, 0x72, 0xe7, 0x17, 0xb0, 0xe5, 0xe5, 0xb7,
        0x20, 0xa5, 0xcb, 0xd9, 0x68, 0x99, 0xad, 0x3f,
        0xde, 0xf3, 0xde, 0x1d, 0x1c, 0x00, 0x74, 0xd2,
        0xd1, 0x57, 0x55, 0x5d, 0xce, 0x76, 0x0c, 0xc4,
        0x7a, 0xc4, 0x65, 0x7c, 0x19, 0x17, 0x0a, 0x09,
        0x66, 0x7d, 0x3a, 0xab, 0xf7, 0x61, 0x3a, 0xe3,
        0x5b, 0xac, 0xcf, 0x69, 0xb0, 0x8b, 0xee, 0x5d,
        0x28, 0x36, 0xbb, 0x3f, 0x74, 0xce, 0x6e, 0x38,
        0x1e, 0x39, 0xab, 0x26, 0xca, 0x89, 0xdc, 0x58,
        0x59, 0xcb, 0x95, 0xe4, 0xbc, 0xd6, 0x19, 0x48,
        0xd0, 0x55, 0x68, 0x7b, 0xb4, 0x27, 0x95, 0x3c,
        0xd9, 0x58, 0x10, 0x4f, 0x8f, 0x55, 0x1c, 0x3f,
        0x04, 0xce, 0x89, 0x1f, 0x82, 0x28, 0xe9, 0x48,
        0x17, 0x47, 0x8f, 0xee, 0xb7, 0x8f, 0xeb, 0xb1,
        0x29, 0xa8, 0x23, 0x18, 0x73, 0x33, 0x9f, 0x83,
        0x08, 0xca, 0xcd, 0x54, 0x6e, 0xca, 0xec, 0x78,
        0x7b, 0x16, 0x83, 0x3f, 0xdb, 0x0a, 0xef, 0xfd,
        0x87, 0x94, 0x19, 0x08, 0x6e, 0x6e, 0x22, 0x57,
        0xd7, 0xd2, 0x79, 0xf9, 0xf6, 0xeb, 0xe0, 0x6c,
        0x93, 0x9d, 0x95, 0xfa, 0x41, 0x7a, 0xa9, 0xd6,
        0x2a, 0xa3, 0x26, 0x9b, 0x24, 0x1b, 0x8b, 0xa0,
        0xed, 0x04, 0xb2, 0xe4, 0x6c, 0x4e, 0xc4, 0x3f,
        0x61, 0xe5, 0xe0, 0x4d, 0x09, 0x28, 0xaf, 0x58,
        0x35, 0x25, 0x0b, 0xd5, 0x38, 0x18, 0x69, 0x51,
        0x18, 0x51, 0x73, 0x7b, 0x28, 0x19, 0x9f, 0xe4,
        0x69, 0xfc, 0x2c, 0x25, 0x08, 0x99, 0x8f, 0x62,
        0x65, 0x62, 0xa5, 0x28, 0xf1, 0xf4, 0xfb, 0x02,
        0x29, 0x27, 0xb0, 0x5e, 0xbb, 0x4f, 0xf9, 0x1a,
        0xa7, 0xc4, 0x38, 0x63, 0x5b, 0x01, 0xfe, 0x00,
        0x66, 0xe3, 0x47, 0x77, 0x21, 0x85, 0x17, 0xd5,
        0x34, 0x19, 0xd3, 0x87, 0xab, 0x44, 0x62, 0x08,
        0x59, 0xb2, 0x6b, 0x1f, 0x21, 0x0c, 0x23, 0x84,
        0xf7, 0xba, 0x92, 0x67, 0xf9, 0x16, 0x85, 0x6a,
        0xe0, 0xeb, 0xe7, 0x4f, 0x06, 0x80, 0x81, 0x81,
        0x28, 0x9c, 0xe8, 0x2e, 0x71, 0x97, 0x48, 0xe0,
        0xd1, 0xbc, 0xce, 0xe9, 0x42, 0x2c, 0x89, 0xdf,
        0x0b, 0xa9, 0xa1, 0x07, 0x84, 0x33, 0x78, 0x7f,
        0x49, 0x2f, 0x1c, 0x55, 0xc3, 0x7f, 0xc3, 0x37,
        0x40, 0xdf, 0x13, 0xf4, 0xa0, 0x21, 0x79, 0x6e,
        0x3a, 0xe3, 0xb8, 0x23, 0x9e, 0x8a, 0x6e, 0x9c
    };

    rc = linuxkm_test_kpp_driver(WOLFKM_FFDHE3072_DRIVER,
                                 b_pub, expected_a_pub, sizeof(b_pub),
                                 secret, sizeof(secret),
                                 shared_secret, sizeof(shared_secret));

    return rc;
}
#endif /* HAVE_FFDHE_3072 */

#ifdef HAVE_FFDHE_4096
static int linuxkm_test_ffdhe4096(void)
{
    int rc = 0;

    /* reference values from kernel crypto/testmgr.h */
    /* 528 byte secret size */
    static const byte secret[] = {
#ifdef LITTLE_ENDIAN_ORDER
        0x01, 0x00, /* type */
        0x10, 0x02, /* len */
        0x00, 0x02, 0x00, 0x00, /* key_size */
        0x00, 0x00, 0x00, 0x00, /* p_size */
        0x00, 0x00, 0x00, 0x00, /* g_size */
#else
        0x00, 0x01, /* type */
        0x02, 0x10, /* len */
        0x00, 0x00, 0x02, 0x00, /* key_size */
        0x00, 0x00, 0x00, 0x00, /* p_size */
        0x00, 0x00, 0x00, 0x00, /* g_size */
#endif
        /* xa */
        0x1a, 0x48, 0xf3, 0x6c, 0x61, 0x03, 0x42, 0x43,
        0xd7, 0x42, 0x3b, 0xfa, 0xdb, 0x55, 0x6f, 0xa2,
        0xe1, 0x79, 0x52, 0x0b, 0x47, 0xc5, 0x03, 0x60,
        0x2f, 0x26, 0xb9, 0x1a, 0x14, 0x15, 0x1a, 0xd9,
        0xe0, 0xbb, 0xa7, 0x82, 0x63, 0x41, 0xec, 0x26,
        0x55, 0x00, 0xab, 0xe5, 0x21, 0x9d, 0x31, 0x14,
        0x0e, 0xe2, 0xc2, 0xb2, 0xb8, 0x37, 0xe6, 0xc3,
        0x5a, 0xab, 0xae, 0x25, 0xdb, 0x71, 0x1e, 0xed,
        0xe8, 0x75, 0x9a, 0x04, 0xa7, 0x92, 0x2a, 0x99,
        0x7e, 0xc0, 0x5b, 0x64, 0x75, 0x7f, 0xe5, 0xb5,
        0xdb, 0x6c, 0x95, 0x4f, 0xe9, 0xdc, 0x39, 0x76,
        0x79, 0xb0, 0xf7, 0x00, 0x30, 0x8e, 0x86, 0xe7,
        0x36, 0xd1, 0xd2, 0x0c, 0x68, 0x7b, 0x94, 0xe9,
        0x91, 0x85, 0x08, 0x86, 0xbc, 0x64, 0x87, 0xd2,
        0xf5, 0x5b, 0xaf, 0x03, 0xf6, 0x5f, 0x28, 0x25,
        0xf1, 0xa3, 0x20, 0x5c, 0x1b, 0xb5, 0x26, 0x45,
        0x9a, 0x47, 0xab, 0xd6, 0xad, 0x49, 0xab, 0x92,
        0x8e, 0x62, 0x6f, 0x48, 0x31, 0xea, 0xf6, 0x76,
        0xff, 0xa2, 0xb6, 0x28, 0x78, 0xef, 0x59, 0xc3,
        0x71, 0x5d, 0xa8, 0xd9, 0x70, 0x89, 0xcc, 0xe2,
        0x63, 0x58, 0x5e, 0x3a, 0xa2, 0xa2, 0x88, 0xbf,
        0x77, 0x20, 0x84, 0x33, 0x65, 0x64, 0x4e, 0x73,
        0xe5, 0x08, 0xd5, 0x89, 0x23, 0xd6, 0x07, 0xac,
        0x29, 0x65, 0x2e, 0x02, 0xa8, 0x35, 0x96, 0x48,
        0xe7, 0x5d, 0x43, 0x6a, 0x42, 0xcc, 0xda, 0x98,
        0xc4, 0x75, 0x90, 0x2e, 0xf6, 0xc4, 0xbf, 0xd4,
        0xbc, 0x31, 0x14, 0x0d, 0x54, 0x30, 0x11, 0xb2,
        0xc9, 0xcf, 0xbb, 0xba, 0xbc, 0xc6, 0xf2, 0xcf,
        0xfe, 0x4a, 0x9d, 0xf3, 0xec, 0x78, 0x5d, 0x5d,
        0xb4, 0x99, 0xd0, 0x67, 0x0f, 0x5a, 0x21, 0x1c,
        0x7b, 0x95, 0x2b, 0xcf, 0x49, 0x44, 0x94, 0x05,
        0x1a, 0x21, 0x81, 0x25, 0x7f, 0xe3, 0x8a, 0x2a,
        0xdd, 0x88, 0xac, 0x44, 0x94, 0x23, 0x20, 0x3b,
        0x75, 0xf6, 0x2a, 0x8a, 0x45, 0xf8, 0xb5, 0x1f,
        0xb9, 0x8b, 0xeb, 0xab, 0x9b, 0x38, 0x23, 0x26,
        0xf1, 0x0f, 0x34, 0x47, 0x4f, 0x7f, 0xe1, 0x9e,
        0x84, 0x84, 0x78, 0xe5, 0xe3, 0x49, 0xeb, 0xcc,
        0x2f, 0x02, 0x85, 0xa4, 0x18, 0x91, 0xde, 0x1a,
        0x60, 0x54, 0x33, 0x81, 0xd5, 0xae, 0xdb, 0x23,
        0x9c, 0x4d, 0xa4, 0xdb, 0x22, 0x5b, 0xdf, 0xf4,
        0x8e, 0x05, 0x2b, 0x60, 0xba, 0xe8, 0x75, 0xfc,
        0x34, 0x99, 0xcf, 0x35, 0xe1, 0x06, 0xba, 0xdc,
        0x79, 0x2a, 0x5e, 0xec, 0x1c, 0xbe, 0x79, 0x33,
        0x63, 0x1c, 0xe7, 0x5f, 0x1e, 0x30, 0xd6, 0x1b,
        0xdb, 0x11, 0xb8, 0xea, 0x63, 0xff, 0xfe, 0x1a,
        0x3c, 0x24, 0xf4, 0x78, 0x9c, 0xcc, 0x5d, 0x9a,
        0xc9, 0x2d, 0xc4, 0x9a, 0xd4, 0xa7, 0x65, 0x84,
        0x98, 0xdb, 0x66, 0x76, 0xf0, 0x34, 0x31, 0x9f,
        0xce, 0xb5, 0xfb, 0x28, 0x07, 0xde, 0x1e, 0x0d,
        0x9b, 0x01, 0x64, 0xeb, 0x2a, 0x37, 0x2f, 0x20,
        0xa5, 0x95, 0x72, 0x2b, 0x54, 0x51, 0x59, 0x91,
        0xea, 0x50, 0x54, 0x0f, 0x2e, 0xb0, 0x1d, 0xf6,
        0xb9, 0x46, 0x43, 0xf9, 0xd0, 0x13, 0x21, 0x20,
        0x47, 0x61, 0x1a, 0x1c, 0x30, 0xc6, 0x9e, 0x75,
        0x22, 0xe4, 0xf2, 0xb1, 0xab, 0x01, 0xdc, 0x5b,
        0x3c, 0x1e, 0xa2, 0x6d, 0xc0, 0xb9, 0x9a, 0x2a,
        0x84, 0x61, 0xea, 0x85, 0x63, 0xa0, 0x77, 0xd0,
        0xeb, 0x20, 0x68, 0xd5, 0x95, 0x6a, 0x1b, 0x8f,
        0x1f, 0x9a, 0xba, 0x44, 0x49, 0x8c, 0x77, 0xa6,
        0xd9, 0xa0, 0x14, 0xf8, 0x7d, 0x9b, 0x4e, 0xfa,
        0xdc, 0x4f, 0x1c, 0x4d, 0x60, 0x50, 0x26, 0x7f,
        0xd6, 0xc1, 0x91, 0x2b, 0xa6, 0x37, 0x5d, 0x94,
        0x69, 0xb2, 0x47, 0x59, 0xd6, 0xc3, 0x59, 0xbb,
        0xd6, 0x9b, 0x71, 0x52, 0x85, 0x7a, 0xcb, 0x2d
    };

    /* 512 byte pub key */
    static const byte b_pub[] = {
        0x24, 0x38, 0x02, 0x02, 0x2f, 0xeb, 0x54, 0xdd,
        0x73, 0x21, 0x91, 0x4a, 0xd8, 0xa4, 0x0a, 0xbf,
        0xf4, 0xf5, 0x9a, 0x45, 0xb5, 0xcd, 0x42, 0xa3,
        0x57, 0xcc, 0x65, 0x4a, 0x23, 0x2e, 0xee, 0x59,
        0xba, 0x6f, 0x14, 0x89, 0xae, 0x2e, 0x14, 0x0a,
        0x72, 0x77, 0x23, 0x7f, 0x6c, 0x2e, 0xba, 0x52,
        0x3f, 0x71, 0xbf, 0xe4, 0x60, 0x03, 0x16, 0xaa,
        0x61, 0xf5, 0x80, 0x1d, 0x8a, 0x45, 0x9e, 0x53,
        0x7b, 0x07, 0xd9, 0x7e, 0xfe, 0xaf, 0xcb, 0xda,
        0xff, 0x20, 0x71, 0xba, 0x89, 0x39, 0x75, 0xc3,
        0xb3, 0x65, 0x0c, 0xb1, 0xa7, 0xfa, 0x4a, 0xe7,
        0xe0, 0x85, 0xc5, 0x4e, 0x91, 0x47, 0x41, 0xf4,
        0xdd, 0xcd, 0xc5, 0x3d, 0x17, 0x12, 0xed, 0xee,
        0xc0, 0x31, 0xb1, 0xaf, 0xc1, 0xd5, 0x3c, 0x07,
        0xa1, 0x5a, 0xc4, 0x05, 0x45, 0xe3, 0x10, 0x0c,
        0xc3, 0x14, 0xae, 0x65, 0xca, 0x40, 0xae, 0x31,
        0x5c, 0x13, 0x0d, 0x32, 0x85, 0xa7, 0x6e, 0xf4,
        0x5e, 0x29, 0x3d, 0x4e, 0xd3, 0xd7, 0x49, 0x58,
        0xe1, 0x73, 0xbb, 0x0a, 0x7b, 0xd6, 0x13, 0xea,
        0x49, 0xd7, 0x20, 0x3d, 0x31, 0xaa, 0x77, 0xab,
        0x21, 0x74, 0xe9, 0x2f, 0xe9, 0x5e, 0xbe, 0x2f,
        0xb4, 0xa2, 0x79, 0xf2, 0xbc, 0xcc, 0x51, 0x94,
        0xd2, 0x1d, 0xb2, 0xe6, 0xc5, 0x39, 0x66, 0xd7,
        0xe5, 0x46, 0x75, 0x53, 0x76, 0xed, 0x49, 0xea,
        0x3b, 0xdd, 0x01, 0x27, 0xdb, 0x83, 0xa5, 0x9f,
        0xd2, 0xee, 0xc8, 0xde, 0x9e, 0xde, 0xd2, 0xe7,
        0x99, 0xad, 0x9c, 0xe0, 0x71, 0x66, 0x29, 0xd8,
        0x0d, 0xfe, 0xdc, 0xd1, 0xbc, 0xc7, 0x9a, 0xbe,
        0x8b, 0x26, 0x46, 0x57, 0xb6, 0x79, 0xfa, 0xad,
        0x8b, 0x45, 0x2e, 0xb5, 0xe5, 0x89, 0x34, 0x01,
        0x93, 0x00, 0x9d, 0xe9, 0x58, 0x74, 0x8b, 0xda,
        0x07, 0x92, 0xb5, 0x01, 0x4a, 0xe1, 0x44, 0x36,
        0xc7, 0x6c, 0xde, 0xc8, 0x7a, 0x17, 0xd0, 0xde,
        0xee, 0x68, 0x92, 0xb5, 0xde, 0x21, 0x2b, 0x1c,
        0xbc, 0x65, 0x30, 0x1e, 0xae, 0x15, 0x3d, 0x9a,
        0xaf, 0x20, 0xa3, 0xc4, 0x21, 0x70, 0xfb, 0x2f,
        0x36, 0x72, 0x31, 0xc0, 0xe8, 0x85, 0xdf, 0xc5,
        0x50, 0x4c, 0x90, 0x10, 0x32, 0xa4, 0xc7, 0xee,
        0x59, 0x5a, 0x21, 0xf4, 0xf1, 0x33, 0xcf, 0xbe,
        0xac, 0x67, 0xb1, 0x40, 0x7c, 0x0b, 0x3f, 0x64,
        0xe5, 0xd2, 0x2d, 0xb7, 0x7d, 0x0f, 0xce, 0xf7,
        0x9b, 0x05, 0xee, 0x37, 0x61, 0xd2, 0x61, 0x9e,
        0x1a, 0x80, 0x2e, 0x79, 0xe6, 0x1b, 0x25, 0xb3,
        0x61, 0x3d, 0x53, 0xe7, 0xe5, 0x97, 0x9a, 0xc2,
        0x39, 0xb1, 0xe3, 0x91, 0xc6, 0xee, 0x96, 0x2e,
        0xa9, 0xb4, 0xb8, 0xad, 0xd8, 0x04, 0x3e, 0x11,
        0x31, 0x67, 0xb8, 0x6a, 0xcb, 0x6e, 0x1a, 0x4c,
        0x7f, 0x74, 0xc7, 0x1f, 0x09, 0xd1, 0xd0, 0x6b,
        0x17, 0xde, 0xea, 0xe8, 0x0b, 0xe6, 0x6a, 0xee,
        0x2f, 0xe3, 0x5b, 0x9c, 0x59, 0x5d, 0x00, 0x57,
        0xbf, 0x24, 0x25, 0xba, 0x22, 0x34, 0xb9, 0xc5,
        0x3c, 0xc4, 0x57, 0x26, 0xd0, 0x6d, 0x89, 0xee,
        0x67, 0x79, 0x3c, 0x70, 0xf9, 0xc3, 0xb4, 0x30,
        0xf0, 0x2e, 0xca, 0xfa, 0x74, 0x00, 0xd1, 0x00,
        0x6d, 0x03, 0x97, 0xd5, 0x08, 0x3f, 0x0b, 0x8e,
        0xb8, 0x1d, 0xa3, 0x91, 0x7f, 0xa9, 0x3a, 0xf0,
        0x37, 0x57, 0x46, 0x87, 0x82, 0xa3, 0xb5, 0x8f,
        0x51, 0xaa, 0xc7, 0x7b, 0xfe, 0x86, 0x26, 0xb9,
        0xfa, 0xe6, 0x1e, 0xee, 0x92, 0x9d, 0x3a, 0xed,
        0x5b, 0x5e, 0x3f, 0xe5, 0xca, 0x5e, 0x13, 0x01,
        0xdd, 0x4c, 0x8d, 0x85, 0xf0, 0x60, 0x61, 0xb7,
        0x60, 0x24, 0x83, 0x9f, 0xbe, 0x72, 0x21, 0x81,
        0x55, 0x7e, 0x7e, 0x6d, 0xf3, 0x28, 0xc8, 0x77,
        0x5a, 0xae, 0x5a, 0x32, 0x86, 0xd5, 0x61, 0xad
    };

    /* 512 byte pub key */
    static const byte expected_a_pub[] = {
        0x1f, 0xff, 0xd6, 0xc4, 0x59, 0xf3, 0x4a, 0x9e,
        0x81, 0x74, 0x4d, 0x27, 0xa7, 0xc6, 0x6b, 0x35,
        0xd8, 0xf5, 0xb3, 0x24, 0x97, 0x82, 0xe7, 0x2e,
        0xf3, 0x21, 0x91, 0x23, 0x2f, 0x3d, 0x57, 0x7f,
        0x15, 0x8c, 0x84, 0x71, 0xe7, 0x25, 0x35, 0xe8,
        0x07, 0x14, 0x06, 0x4c, 0x83, 0xdc, 0x55, 0x4a,
        0xf8, 0x45, 0xc5, 0xe9, 0xfa, 0x6e, 0xae, 0x6e,
        0xcf, 0x4d, 0x11, 0x91, 0x26, 0x16, 0x6f, 0x86,
        0x89, 0x78, 0xaa, 0xb4, 0x25, 0x54, 0xb2, 0x74,
        0x07, 0xe5, 0x26, 0x26, 0x0c, 0xad, 0xa4, 0x57,
        0x59, 0x61, 0x66, 0x71, 0x43, 0x22, 0xff, 0x49,
        0x51, 0xa4, 0x76, 0x0e, 0x55, 0x7b, 0x60, 0x45,
        0x4f, 0xaf, 0xbd, 0x9c, 0xec, 0x64, 0x3f, 0x80,
        0x0b, 0x0c, 0x31, 0x41, 0xf0, 0xfe, 0x2c, 0xb7,
        0x0a, 0xbe, 0xa5, 0x71, 0x08, 0x0d, 0x8d, 0x1e,
        0x8a, 0x77, 0x9a, 0xd2, 0x90, 0x31, 0x96, 0xd0,
        0x3b, 0x31, 0xdc, 0xc6, 0x18, 0x59, 0x43, 0xa1,
        0x19, 0x5a, 0x84, 0x68, 0x29, 0xad, 0x5e, 0x58,
        0xa2, 0x50, 0x3e, 0x83, 0xf5, 0x7a, 0xbd, 0x88,
        0x17, 0x60, 0x89, 0x98, 0x9c, 0x19, 0x89, 0x27,
        0x89, 0xfc, 0x33, 0x87, 0x42, 0xd5, 0xde, 0x19,
        0x14, 0xf2, 0x95, 0x82, 0x10, 0x87, 0xad, 0x82,
        0xdd, 0x6b, 0x51, 0x2d, 0x8d, 0x0e, 0x81, 0x4b,
        0xde, 0xb3, 0x35, 0x6c, 0x0f, 0x4b, 0x56, 0x45,
        0x48, 0x87, 0xe9, 0x5a, 0xf9, 0x70, 0x10, 0x30,
        0x8e, 0xa1, 0xbb, 0xa4, 0x70, 0xbf, 0xa0, 0xab,
        0x10, 0x31, 0x3c, 0x2c, 0xdc, 0xc4, 0xed, 0xe3,
        0x51, 0xdc, 0xee, 0xd2, 0xa5, 0x5c, 0x4e, 0x6e,
        0xf6, 0xed, 0x60, 0x5a, 0xeb, 0xf3, 0x02, 0x19,
        0x2a, 0x95, 0xe9, 0x46, 0xff, 0x37, 0x1b, 0xf0,
        0x1d, 0x10, 0x4a, 0x8f, 0x4f, 0x3a, 0x6e, 0xf5,
        0xfc, 0x02, 0x6d, 0x09, 0x7d, 0xea, 0x69, 0x7b,
        0x13, 0xb0, 0xb6, 0x80, 0x5c, 0x15, 0x20, 0xa8,
        0x4d, 0x15, 0x56, 0x11, 0x72, 0x49, 0xdb, 0x48,
        0x54, 0x40, 0x66, 0xd5, 0xcd, 0x17, 0x3a, 0x26,
        0x95, 0xf6, 0xd7, 0xf2, 0x59, 0xa3, 0xda, 0xbb,
        0x26, 0xd0, 0xe5, 0x46, 0xbf, 0xee, 0x0e, 0x7d,
        0xf1, 0xe0, 0x11, 0x02, 0x4d, 0xd3, 0xdc, 0xe2,
        0x3f, 0xc2, 0x51, 0x7e, 0xc7, 0x90, 0x33, 0x3c,
        0x1c, 0xa0, 0x4c, 0x69, 0xcc, 0x1e, 0xc7, 0xac,
        0x17, 0xe0, 0xe5, 0xf4, 0x8c, 0x05, 0x64, 0x34,
        0xfe, 0x84, 0x70, 0xd7, 0x6b, 0xed, 0xab, 0xf5,
        0x88, 0x9d, 0x3e, 0x4c, 0x5a, 0x9e, 0xd4, 0x74,
        0xfd, 0xdd, 0x91, 0xd5, 0xd4, 0xcb, 0xbf, 0xf8,
        0xb7, 0x56, 0xb5, 0xe9, 0x22, 0xa6, 0x6d, 0x7a,
        0x44, 0x05, 0x41, 0xbf, 0xdb, 0x61, 0x28, 0xc6,
        0x99, 0x49, 0x87, 0x3d, 0x28, 0x77, 0xf8, 0x83,
        0x23, 0x7e, 0xa9, 0xa7, 0xee, 0x20, 0xdb, 0x6d,
        0x21, 0x50, 0xb7, 0xc9, 0x52, 0x57, 0x53, 0xa3,
        0xcf, 0xdf, 0xd0, 0xf9, 0xb9, 0x62, 0x96, 0x89,
        0xf5, 0x5c, 0xa9, 0x8a, 0x11, 0x95, 0x01, 0x25,
        0xc9, 0x81, 0x15, 0x76, 0xae, 0xf0, 0xc7, 0xc5,
        0x50, 0xae, 0x6f, 0xb5, 0xd2, 0x8a, 0x8e, 0x9a,
        0xd4, 0x30, 0x55, 0xc6, 0xe9, 0x2c, 0x81, 0x6e,
        0x95, 0xf6, 0x45, 0x89, 0x55, 0x28, 0x34, 0x7b,
        0xe5, 0x72, 0x9a, 0x2a, 0xe2, 0x98, 0x09, 0x35,
        0xe0, 0xe9, 0x75, 0x94, 0xe9, 0x34, 0x95, 0xb9,
        0x13, 0x6e, 0xd5, 0xa1, 0x62, 0x5a, 0x1c, 0x94,
        0x28, 0xed, 0x84, 0x46, 0x76, 0x6d, 0x10, 0x37,
        0x71, 0xa3, 0x31, 0x46, 0x64, 0xe4, 0x59, 0x44,
        0x17, 0x70, 0x1c, 0x23, 0xc9, 0x7e, 0xf6, 0xab,
        0x8a, 0x24, 0xae, 0x25, 0xe2, 0xb2, 0x5f, 0x33,
        0xe4, 0xd7, 0xd3, 0x34, 0x2a, 0x49, 0x22, 0x16,
        0x15, 0x9b, 0x90, 0x40, 0xda, 0x99, 0xd5, 0xaf
    };

    /* 512 byte shared secret */
    static const byte shared_secret[] = {
        0xe2, 0xce, 0x0e, 0x4b, 0x64, 0xf3, 0x84, 0x62,
        0x38, 0xfd, 0xe3, 0x6f, 0x69, 0x40, 0x22, 0xb0,
        0x73, 0x27, 0x03, 0x12, 0x82, 0xa4, 0x6e, 0x03,
        0x57, 0xec, 0x3d, 0xa0, 0xc1, 0x4f, 0x4b, 0x09,
        0xa1, 0xd4, 0xe0, 0x1a, 0x5d, 0x91, 0x2e, 0x08,
        0xad, 0x57, 0xfa, 0xcc, 0x55, 0x90, 0x5f, 0xa0,
        0x52, 0x27, 0x62, 0x8d, 0xe5, 0x2d, 0xa1, 0x5f,
        0xf0, 0x30, 0x43, 0x77, 0x4e, 0x3f, 0x02, 0x58,
        0xcb, 0xa0, 0x51, 0xae, 0x1d, 0x24, 0xf9, 0x0a,
        0xd1, 0x36, 0x0b, 0x95, 0x0f, 0x07, 0xd9, 0xf7,
        0xe2, 0x36, 0x14, 0x2f, 0xf0, 0x11, 0xc2, 0xc9,
        0xaf, 0x66, 0x4e, 0x0d, 0xb4, 0x60, 0x01, 0x4e,
        0xa8, 0x49, 0xc6, 0xec, 0x5f, 0xb2, 0xbc, 0x05,
        0x48, 0x91, 0x4e, 0xe1, 0xc3, 0x99, 0x9f, 0xeb,
        0x4a, 0xc1, 0xde, 0x05, 0x9a, 0x65, 0x39, 0x7d,
        0x2f, 0x89, 0x85, 0xb2, 0xcf, 0xec, 0x25, 0x27,
        0x5f, 0x1c, 0x11, 0x63, 0xcf, 0x7b, 0x86, 0x98,
        0x39, 0xae, 0xc2, 0x16, 0x8f, 0x79, 0xd1, 0x20,
        0xd0, 0xb4, 0xa0, 0xba, 0x44, 0xd8, 0xf5, 0x3a,
        0x0a, 0x08, 0x4c, 0xd1, 0xb9, 0xdd, 0x0a, 0x5b,
        0x9e, 0x62, 0xf3, 0x52, 0x0c, 0x84, 0x12, 0x43,
        0x9b, 0xd7, 0xdf, 0x86, 0x71, 0x03, 0xdd, 0x04,
        0x98, 0x55, 0x0c, 0x7b, 0xe2, 0xe8, 0x03, 0x17,
        0x25, 0x84, 0xd9, 0xbd, 0xe1, 0xce, 0x64, 0xbe,
        0xca, 0x55, 0xd4, 0x5b, 0xef, 0x61, 0x5b, 0x68,
        0x4b, 0x80, 0x37, 0x40, 0xae, 0x28, 0x87, 0x81,
        0x55, 0x34, 0x96, 0x50, 0x21, 0x47, 0x49, 0xc0,
        0xda, 0x26, 0x46, 0xb8, 0xe8, 0xcc, 0x5a, 0x27,
        0x9c, 0x9d, 0x0a, 0x3d, 0xcc, 0x4c, 0x63, 0x27,
        0x81, 0x82, 0x2e, 0xf4, 0xa8, 0x91, 0x37, 0x3e,
        0xa7, 0x34, 0x6a, 0x0f, 0x60, 0x44, 0xdd, 0x2e,
        0xdc, 0xf9, 0x19, 0xf2, 0x2e, 0x81, 0x05, 0x51,
        0x16, 0xbc, 0xc0, 0x85, 0xa5, 0xd5, 0x08, 0x09,
        0x1f, 0xcd, 0xed, 0xa4, 0xc5, 0xdb, 0x16, 0x43,
        0xb5, 0x7a, 0x71, 0x66, 0x19, 0x2e, 0xef, 0x13,
        0xbc, 0x40, 0x39, 0x0a, 0x00, 0x45, 0x7e, 0x61,
        0xe9, 0x68, 0x60, 0x83, 0x00, 0x70, 0xd1, 0x71,
        0xd3, 0xa2, 0x61, 0x3e, 0x00, 0x46, 0x93, 0x0d,
        0xbf, 0xe6, 0xa2, 0x07, 0xe6, 0x40, 0x1a, 0xf4,
        0x57, 0xc6, 0x67, 0x39, 0xd8, 0xd7, 0x6b, 0xc5,
        0xa5, 0xd8, 0x38, 0x78, 0x12, 0xb4, 0x97, 0x12,
        0xbe, 0x97, 0x13, 0xef, 0xe4, 0x74, 0x0c, 0xe0,
        0x75, 0x89, 0x64, 0xf4, 0xe8, 0x85, 0xda, 0x84,
        0x7b, 0x1d, 0xfe, 0xdd, 0x21, 0xba, 0xda, 0x01,
        0x52, 0xdc, 0x59, 0xe5, 0x47, 0x50, 0x7e, 0x15,
        0x20, 0xd0, 0x43, 0x37, 0x6e, 0x48, 0x39, 0x00,
        0xee, 0xd9, 0x54, 0x6d, 0x00, 0x65, 0xc9, 0x4b,
        0x85, 0xa2, 0x8a, 0x40, 0x55, 0xd0, 0x63, 0x0c,
        0xb5, 0x7a, 0x0d, 0x37, 0x67, 0x27, 0x73, 0x18,
        0x7f, 0x5a, 0xf5, 0x0e, 0x22, 0xb9, 0xb0, 0x3f,
        0xda, 0xf1, 0xec, 0x7c, 0x24, 0x01, 0x49, 0xa9,
        0x09, 0x0e, 0x0f, 0xc4, 0xa9, 0xef, 0xc8, 0x2b,
        0x13, 0xd1, 0x0a, 0x6f, 0xf8, 0x92, 0x4b, 0x1d,
        0xdd, 0x6c, 0x9c, 0x35, 0xde, 0x75, 0x46, 0x32,
        0xe6, 0xfb, 0xda, 0x58, 0xba, 0x81, 0x08, 0xca,
        0xa9, 0xb6, 0x69, 0x71, 0x96, 0x2a, 0x1f, 0x2e,
        0x25, 0xe0, 0x37, 0xfe, 0xee, 0x4d, 0x27, 0xaa,
        0x04, 0xda, 0x95, 0xbb, 0x93, 0xcf, 0x8f, 0xa2,
        0x1d, 0x67, 0x35, 0xe3, 0x51, 0x8f, 0x87, 0x3b,
        0xa9, 0x62, 0x05, 0xee, 0x44, 0xb7, 0x2e, 0xd0,
        0x07, 0x63, 0x32, 0xf5, 0xcd, 0x64, 0x18, 0x20,
        0xcf, 0x22, 0x42, 0x28, 0x22, 0x1a, 0xa8, 0xbb,
        0x74, 0x8a, 0x6f, 0x2a, 0xea, 0x8a, 0x48, 0x0a,
        0xad, 0xd7, 0xed, 0xba, 0xa3, 0x89, 0x37, 0x01
    };

    rc = linuxkm_test_kpp_driver(WOLFKM_FFDHE4096_DRIVER,
                                 b_pub, expected_a_pub, sizeof(b_pub),
                                 secret, sizeof(secret),
                                 shared_secret, sizeof(shared_secret));

    return rc;
}
#endif /* HAVE_FFDHE_4096 */

#ifdef HAVE_FFDHE_6144
static int linuxkm_test_ffdhe6144(void)
{
    int rc = 0;

    /* reference values from kernel crypto/testmgr.h */
    /* 784 byte secret size */
    static const byte secret[] = {
#ifdef LITTLE_ENDIAN_ORDER
        0x01, 0x00, /* type */
        0x10, 0x03, /* len */
        0x00, 0x03, 0x00, 0x00, /* key_size */
        0x00, 0x00, 0x00, 0x00, /* p_size */
        0x00, 0x00, 0x00, 0x00, /* g_size */
#else
        0x00, 0x01, /* type */
        0x03, 0x10, /* len */
        0x00, 0x00, 0x03, 0x00, /* key_size */
        0x00, 0x00, 0x00, 0x00, /* p_size */
        0x00, 0x00, 0x00, 0x00, /* g_size */
#endif
        /* xa */
        0x63, 0x3e, 0x6f, 0xe0, 0xfe, 0x9f, 0x4a, 0x01,
        0x62, 0x77, 0xce, 0xf1, 0xc7, 0xcc, 0x49, 0x4d,
        0x92, 0x53, 0x56, 0xe3, 0x39, 0x15, 0x81, 0xb2,
        0xcd, 0xdc, 0xaf, 0x5e, 0xbf, 0x31, 0x1f, 0x69,
        0xce, 0x41, 0x35, 0x24, 0xaa, 0x46, 0x53, 0xb5,
        0xb7, 0x3f, 0x2b, 0xad, 0x95, 0x14, 0xfb, 0xe4,
        0x9a, 0x61, 0xcd, 0x0f, 0x1f, 0x02, 0xee, 0xa4,
        0x79, 0x2c, 0x9d, 0x1a, 0x7c, 0x62, 0x82, 0x39,
        0xdd, 0x43, 0xcc, 0x58, 0x9f, 0x62, 0x47, 0x56,
        0x1d, 0x0f, 0xc2, 0x67, 0xbc, 0x24, 0xd0, 0xf9,
        0x0a, 0x50, 0x1b, 0x10, 0xe7, 0xbb, 0xd1, 0xc2,
        0x01, 0xbb, 0xc4, 0x4c, 0xda, 0x12, 0x60, 0x0e,
        0x95, 0x2b, 0xde, 0x09, 0xd6, 0x67, 0xe1, 0xbc,
        0x4c, 0xb9, 0x67, 0xdf, 0xd0, 0x1f, 0x97, 0xb4,
        0xde, 0xcb, 0x6b, 0x78, 0x83, 0x51, 0x74, 0x33,
        0x01, 0x7f, 0xf6, 0x0a, 0x95, 0x69, 0x93, 0x00,
        0x2a, 0xc3, 0x75, 0x8e, 0xef, 0xbe, 0x53, 0x11,
        0x6d, 0xc4, 0xd0, 0x9f, 0x6d, 0x63, 0x48, 0xc1,
        0x91, 0x1f, 0x7d, 0x88, 0xa7, 0x90, 0x78, 0xd1,
        0x7e, 0x52, 0x42, 0x10, 0x01, 0xb4, 0x27, 0x95,
        0x91, 0x43, 0xcc, 0x82, 0x91, 0x86, 0x62, 0xa0,
        0x9d, 0xef, 0x65, 0x6e, 0x67, 0xcf, 0x19, 0x11,
        0x35, 0x37, 0x5e, 0x94, 0x97, 0x83, 0xa6, 0x83,
        0x1c, 0x7e, 0x8a, 0x3e, 0x32, 0xb0, 0xce, 0xff,
        0x20, 0xdc, 0x7b, 0x6e, 0x18, 0xd9, 0x6b, 0x27,
        0x31, 0xfc, 0xc3, 0xef, 0x47, 0x8d, 0xbe, 0x34,
        0x2b, 0xc7, 0x60, 0x74, 0x3c, 0x93, 0xb3, 0x8e,
        0x54, 0x77, 0x4e, 0x73, 0xe6, 0x40, 0x72, 0x35,
        0xb0, 0xf0, 0x06, 0x53, 0x43, 0xbe, 0xd0, 0xc3,
        0x87, 0xcc, 0x38, 0x96, 0xa9, 0x10, 0xa0, 0xd6,
        0x17, 0xed, 0xa5, 0x6a, 0xf4, 0xf6, 0xaa, 0x77,
        0x40, 0xed, 0x7d, 0x2e, 0x58, 0x0f, 0x5b, 0x04,
        0x5a, 0x41, 0x12, 0x95, 0x22, 0xcb, 0xa3, 0xce,
        0x8b, 0x6d, 0x6d, 0x89, 0xec, 0x7c, 0x1d, 0x25,
        0x27, 0x52, 0x50, 0xa0, 0x5b, 0x93, 0x8c, 0x5d,
        0x3f, 0x56, 0xb9, 0xa6, 0x5e, 0xe5, 0xf7, 0x9b,
        0xc7, 0x9a, 0x4a, 0x2e, 0x79, 0xb5, 0xca, 0x29,
        0x58, 0x52, 0xa0, 0x63, 0xe4, 0x9d, 0xeb, 0x4c,
        0x4c, 0xa8, 0x37, 0x0b, 0xe9, 0xa0, 0x18, 0xf1,
        0x86, 0xf6, 0x4d, 0x32, 0xfb, 0x9e, 0x4f, 0xb3,
        0x7b, 0x5d, 0x58, 0x78, 0x70, 0xbd, 0x56, 0xac,
        0x99, 0x75, 0x25, 0x71, 0x66, 0x76, 0x4e, 0x5e,
        0x67, 0x4f, 0xb1, 0x17, 0xa7, 0x8b, 0x55, 0x12,
        0x87, 0x01, 0x4e, 0xd1, 0x66, 0xef, 0xd0, 0x70,
        0xaf, 0x14, 0x34, 0xee, 0x2a, 0x76, 0x49, 0x25,
        0xa6, 0x2e, 0x43, 0x37, 0x75, 0x7d, 0x1a, 0xad,
        0x08, 0xd5, 0x01, 0x85, 0x9c, 0xe1, 0x20, 0xd8,
        0x38, 0x5c, 0x57, 0xa5, 0xed, 0x9d, 0x46, 0x3a,
        0xb7, 0x46, 0x60, 0x29, 0x8b, 0xc4, 0x21, 0x50,
        0x0a, 0x30, 0x9c, 0x57, 0x42, 0xe4, 0x35, 0xf8,
        0x12, 0x5c, 0x4f, 0xa2, 0x20, 0xc2, 0xc9, 0x43,
        0xe3, 0x6d, 0x20, 0xbc, 0xdf, 0xb8, 0x37, 0x33,
        0x45, 0x43, 0x06, 0x4e, 0x08, 0x6f, 0x8a, 0xcd,
        0x61, 0xc3, 0x1b, 0x05, 0x28, 0x82, 0xbe, 0xf0,
        0x48, 0x33, 0xe5, 0x93, 0xc9, 0x1a, 0x61, 0x16,
        0x67, 0x03, 0x9d, 0x47, 0x9d, 0x74, 0xeb, 0xae,
        0x13, 0xf2, 0xb4, 0x1b, 0x09, 0x11, 0xf5, 0x15,
        0xcb, 0x28, 0xfd, 0x50, 0xe0, 0xbc, 0x58, 0x36,
        0x38, 0x91, 0x2c, 0x07, 0x27, 0x1f, 0x49, 0x68,
        0xf4, 0xce, 0xad, 0xf7, 0xba, 0xec, 0x5d, 0x3d,
        0xfd, 0x27, 0xe2, 0xcf, 0xf4, 0x56, 0xfe, 0x08,
        0xa6, 0x11, 0x61, 0xcb, 0x6c, 0x9f, 0xf9, 0x3c,
        0x57, 0x0b, 0x8b, 0xaa, 0x00, 0x16, 0x18, 0xba,
        0x1f, 0xe8, 0x4f, 0x01, 0xe2, 0x79, 0x2a, 0x0b,
        0xc1, 0xbd, 0x52, 0xef, 0xe6, 0xf7, 0x5a, 0x66,
        0xfe, 0x07, 0x3b, 0x50, 0x6b, 0xbb, 0xcb, 0x39,
        0x3c, 0x94, 0xf6, 0x21, 0x0d, 0x68, 0x69, 0xa4,
        0xed, 0x2e, 0xb5, 0x85, 0x03, 0x11, 0x38, 0x79,
        0xec, 0xb5, 0x22, 0x23, 0xdf, 0x9e, 0xad, 0xb4,
        0xbe, 0xd7, 0xc7, 0xdf, 0xea, 0x30, 0x23, 0x8a,
        0xb7, 0x21, 0x0a, 0x9d, 0xbd, 0x99, 0x13, 0x7d,
        0x5f, 0x7e, 0xaf, 0x28, 0x54, 0x3f, 0xca, 0x5e,
        0xf4, 0xfc, 0x05, 0x0d, 0x65, 0x67, 0xd8, 0xf6,
        0x8e, 0x90, 0x9d, 0x0d, 0xcf, 0x62, 0x82, 0xd6,
        0x9f, 0x02, 0xf8, 0xca, 0xfa, 0x42, 0x24, 0x7f,
        0x4d, 0xb7, 0xfc, 0x92, 0xa6, 0x4a, 0x51, 0xc4,
        0xd8, 0xae, 0x19, 0x87, 0xc6, 0xa3, 0x83, 0xbe,
        0x7b, 0x6d, 0xc3, 0xf5, 0xb8, 0xad, 0x4a, 0x05,
        0x78, 0x84, 0x3a, 0x15, 0x2e, 0x40, 0xbe, 0x79,
        0xa9, 0xc0, 0x12, 0xa1, 0x48, 0x39, 0xc3, 0xdb,
        0x47, 0x4f, 0x7d, 0xea, 0x6d, 0xc7, 0xfa, 0x2c,
        0x4e, 0xe9, 0xa5, 0x85, 0x81, 0xea, 0x6c, 0xcd,
        0x8a, 0xe5, 0x74, 0x17, 0x76, 0x31, 0x31, 0x75,
        0x96, 0x83, 0xca, 0x81, 0xbb, 0x5c, 0xa9, 0x79,
        0x2c, 0xbd, 0x09, 0xfe, 0xe4, 0x86, 0x0d, 0x8c,
        0x76, 0x9c, 0xbc, 0xe8, 0x93, 0xe4, 0xd0, 0xe4,
        0x0f, 0xf8, 0xff, 0x24, 0x7e, 0x66, 0x61, 0x69,
        0xfb, 0xe4, 0x46, 0x08, 0x94, 0x99, 0xa5, 0x53,
        0xd7, 0xe4, 0x29, 0x72, 0x86, 0x86, 0xe8, 0x1d,
        0x37, 0xfa, 0xcb, 0xd0, 0x8d, 0x51, 0xd0, 0xbf,
        0x81, 0xcf, 0x55, 0xb9, 0xc5, 0x78, 0x8c, 0x74,
        0xa0, 0x16, 0x3a, 0xd2, 0x19, 0x94, 0x29, 0x6a,
        0x5e, 0xec, 0xd3, 0x20, 0xa0, 0xb2, 0xfd, 0xce,
        0xd4, 0x14, 0xa3, 0x39, 0x10, 0xa9, 0xf4, 0x4e,
        0xba, 0x21, 0x09, 0x5c, 0xe6, 0x61, 0x43, 0x51,
        0xae, 0xc4, 0x71, 0xd7, 0x21, 0xef, 0x98, 0x39
    };

    /* 768 byte pub key */
    static const byte b_pub[] = {
        0x30, 0x31, 0xbe, 0x43, 0xd0, 0x14, 0x22, 0x6b,
        0x4b, 0x8c, 0x9a, 0xca, 0xc6, 0xdd, 0xe5, 0x99,
        0xce, 0xb8, 0x30, 0x23, 0xb6, 0xa8, 0x8c, 0x4d,
        0xfa, 0xef, 0xad, 0xa6, 0x6a, 0x21, 0x50, 0xa6,
        0x45, 0x2d, 0x19, 0x2a, 0x29, 0x81, 0xc5, 0xac,
        0xb4, 0xa8, 0x5f, 0x6d, 0x5b, 0xc8, 0x5f, 0x12,
        0x35, 0x21, 0xfb, 0x37, 0xaa, 0x0c, 0x79, 0xeb,
        0xd4, 0x83, 0x01, 0xda, 0xa3, 0xf3, 0x51, 0x6e,
        0x17, 0xf9, 0xef, 0x3f, 0xbd, 0x2f, 0xd2, 0x43,
        0x82, 0x12, 0x48, 0xeb, 0x61, 0x4c, 0x8e, 0xf2,
        0x6c, 0x76, 0xf9, 0x6d, 0x42, 0x2a, 0xcb, 0x10,
        0x13, 0x3b, 0xf6, 0x9b, 0xcd, 0x46, 0x1e, 0xa2,
        0xa7, 0x2c, 0x08, 0x56, 0xd2, 0x42, 0xf5, 0x03,
        0xf0, 0x3e, 0xef, 0xa2, 0xa2, 0xf2, 0x4c, 0xf2,
        0xdb, 0x4f, 0xeb, 0x40, 0x15, 0x53, 0x27, 0xf7,
        0xd4, 0x8e, 0x58, 0x23, 0xf5, 0x2c, 0x88, 0x04,
        0x1e, 0xb1, 0xb6, 0xe3, 0xd6, 0x9c, 0x49, 0x08,
        0xa1, 0x4b, 0xb8, 0x33, 0xe4, 0x75, 0x85, 0xa1,
        0x86, 0x97, 0xce, 0x1d, 0xe9, 0x9f, 0xe2, 0xd8,
        0xf2, 0x7e, 0xad, 0xdc, 0x8a, 0x4d, 0xbd, 0x06,
        0x52, 0x00, 0x9a, 0x2c, 0x69, 0xdd, 0x02, 0x0c,
        0x69, 0x5a, 0xf9, 0x1d, 0xfd, 0xdc, 0xfb, 0x82,
        0xb2, 0xe5, 0xf3, 0x24, 0xba, 0xd1, 0x09, 0x76,
        0x90, 0xb5, 0x7a, 0x92, 0xa6, 0x6b, 0x97, 0xc0,
        0xce, 0x13, 0x9b, 0x4b, 0xbc, 0x30, 0x91, 0xb2,
        0x13, 0x8b, 0x57, 0x6c, 0x8b, 0x66, 0x6e, 0x58,
        0x3e, 0x91, 0x50, 0xc7, 0x6c, 0xe1, 0x18, 0xec,
        0xbf, 0x69, 0xcd, 0xcb, 0xa0, 0xbc, 0x0d, 0x05,
        0xc4, 0xf8, 0x45, 0x92, 0xe0, 0x05, 0xd3, 0x08,
        0xb3, 0x30, 0x19, 0xc8, 0x80, 0xf8, 0x17, 0x9f,
        0x1e, 0x6a, 0x49, 0x8e, 0x43, 0xef, 0x7a, 0x49,
        0xa5, 0x93, 0xd9, 0xed, 0xd1, 0x07, 0x03, 0xe4,
        0xa3, 0x55, 0xeb, 0x1e, 0x2f, 0x69, 0xd7, 0x40,
        0x8f, 0x6e, 0x1c, 0xb6, 0x94, 0xfb, 0xba, 0x4e,
        0x46, 0xd0, 0x38, 0x71, 0x00, 0x88, 0x93, 0x6a,
        0x55, 0xfc, 0x16, 0x95, 0x1f, 0xb1, 0xf6, 0x2f,
        0x26, 0x45, 0x50, 0x54, 0x30, 0x62, 0x62, 0xe8,
        0x80, 0xe5, 0x24, 0x0b, 0xe4, 0x15, 0x6b, 0x32,
        0x16, 0xc2, 0x30, 0x9b, 0x56, 0xb4, 0xc9, 0x5e,
        0x50, 0xb4, 0x27, 0x82, 0x86, 0x01, 0xda, 0x68,
        0x44, 0x4b, 0x15, 0x81, 0x31, 0x13, 0x52, 0xd8,
        0x08, 0xbc, 0xae, 0xf3, 0xa5, 0x94, 0x1c, 0x81,
        0xe8, 0x42, 0xd6, 0x42, 0xd6, 0xff, 0x99, 0x58,
        0x0f, 0x61, 0x3e, 0x82, 0x9e, 0x2d, 0x13, 0x03,
        0x54, 0x02, 0x74, 0xf4, 0x6b, 0x43, 0x43, 0xce,
        0x54, 0x44, 0x36, 0x3f, 0x55, 0xfa, 0xb2, 0x56,
        0xdc, 0xac, 0xb5, 0x65, 0x89, 0xbe, 0x36, 0xd2,
        0x58, 0x65, 0x79, 0x4c, 0xf3, 0xe2, 0x01, 0xf1,
        0x69, 0x96, 0x29, 0x20, 0x5d, 0xee, 0xf5, 0x8a,
        0x8b, 0x9f, 0x72, 0xf7, 0x27, 0x02, 0xde, 0x3b,
        0xc7, 0x52, 0x19, 0xdc, 0x8e, 0x22, 0x36, 0x09,
        0x14, 0x59, 0x07, 0xbb, 0x1e, 0x49, 0x69, 0x4f,
        0x00, 0x7b, 0x9a, 0x5d, 0x23, 0xe9, 0xbe, 0x0d,
        0x52, 0x90, 0xa3, 0x0d, 0xde, 0xe7, 0x80, 0x57,
        0x53, 0x69, 0x39, 0xe6, 0xf8, 0x33, 0xeb, 0x92,
        0x0d, 0x9e, 0x04, 0x8b, 0x16, 0x16, 0x16, 0x1c,
        0xa9, 0xe6, 0xe3, 0x0e, 0x0a, 0xc6, 0xf6, 0x61,
        0xd1, 0x44, 0x2b, 0x3e, 0x5e, 0x02, 0xfe, 0xaa,
        0xe3, 0xf3, 0x8f, 0xf9, 0xc8, 0x20, 0x37, 0xad,
        0xbc, 0x95, 0xb8, 0xc5, 0xe7, 0x95, 0xda, 0xfb,
        0x80, 0x5b, 0xf6, 0x40, 0x28, 0xae, 0xc1, 0x4c,
        0x09, 0xde, 0xff, 0x1e, 0xbf, 0x51, 0xd2, 0xfe,
        0x08, 0xdc, 0xb0, 0x48, 0x21, 0xf5, 0x4c, 0x43,
        0xdc, 0x7b, 0x69, 0x83, 0xc8, 0x69, 0x5c, 0xc4,
        0xa9, 0x98, 0x76, 0x4b, 0xc4, 0x4a, 0xac, 0x1d,
        0xa5, 0x52, 0xe3, 0x35, 0x43, 0xdd, 0x30, 0xd4,
        0xa0, 0x51, 0x9c, 0xc2, 0x62, 0x4c, 0x7e, 0xa5,
        0xfb, 0xd3, 0x2c, 0x8a, 0x09, 0x7f, 0x53, 0xa3,
        0xcd, 0xca, 0x58, 0x1b, 0x4c, 0xaf, 0xba, 0x21,
        0x8b, 0x88, 0x1d, 0xc0, 0xe9, 0x0a, 0x17, 0x30,
        0x33, 0xd6, 0xa2, 0xa5, 0x49, 0x50, 0x61, 0x3b,
        0xff, 0x37, 0x71, 0x66, 0xef, 0x61, 0xbc, 0xb2,
        0x53, 0x82, 0xe5, 0x70, 0xef, 0x32, 0xff, 0x9d,
        0x97, 0xe0, 0x82, 0xe0, 0xbb, 0x49, 0xc2, 0x29,
        0x58, 0x89, 0xdd, 0xe9, 0x62, 0x52, 0xfb, 0xba,
        0x22, 0xa6, 0xd9, 0x16, 0xfa, 0x55, 0xb3, 0x06,
        0xed, 0x6d, 0x70, 0x6e, 0xdc, 0x47, 0x7c, 0x67,
        0x1a, 0xcc, 0x27, 0x98, 0xd4, 0xd7, 0xe6, 0xf0,
        0xf8, 0x9f, 0x51, 0x3e, 0xf0, 0xee, 0xad, 0xb6,
        0x78, 0x69, 0x71, 0xb5, 0xcb, 0x09, 0xa3, 0xa6,
        0x3f, 0x29, 0x24, 0x46, 0xe0, 0x65, 0xbc, 0x9f,
        0x6c, 0xe9, 0xf9, 0x49, 0x49, 0x96, 0x75, 0xe5,
        0xe1, 0xff, 0x82, 0x70, 0xf4, 0x7e, 0xff, 0x8f,
        0xec, 0x47, 0x98, 0x6d, 0x5b, 0x88, 0x60, 0xee,
        0x43, 0xb1, 0xe2, 0x14, 0xc1, 0x49, 0x95, 0x74,
        0x46, 0xd3, 0x3f, 0x73, 0xb2, 0xe9, 0x88, 0xe0,
        0xd3, 0xb1, 0xc4, 0x2c, 0xef, 0xee, 0xdd, 0x6c,
        0xc5, 0xa1, 0x29, 0xef, 0x86, 0xd2, 0x36, 0x8a,
        0x2f, 0x7c, 0x9d, 0x28, 0x0a, 0x6d, 0xc9, 0x5a,
        0xdb, 0xd4, 0x04, 0x06, 0x36, 0x96, 0x09, 0x03,
        0x71, 0x5d, 0x38, 0x67, 0xa2, 0x08, 0x2a, 0x04,
        0xe7, 0xd6, 0x51, 0x5a, 0x19, 0x9d, 0xe7, 0xf1,
        0x5d, 0x6f, 0xe2, 0xff, 0x48, 0x37, 0xb7, 0x8b,
        0xb1, 0x14, 0xb4, 0x96, 0xcd, 0xf0, 0xa7, 0xbd,
        0xef, 0x20, 0xff, 0x0a, 0x8d, 0x08, 0xb7, 0x15,
        0x98, 0x5a, 0x13, 0xd2, 0xda, 0x2a, 0x27, 0x75
    };

    /* 768 byte pub key */
    static const byte expected_a_pub[] = {
        0x45, 0x96, 0x5a, 0xb7, 0x78, 0x5c, 0xa4, 0x4d,
        0x39, 0xb2, 0x5f, 0xc8, 0xc2, 0xaa, 0x1a, 0xf4,
        0xa6, 0x68, 0xf6, 0x6f, 0x7e, 0xa8, 0x4a, 0x5b,
        0x0e, 0xba, 0x0a, 0x99, 0x85, 0xf9, 0x63, 0xd4,
        0x58, 0x21, 0x6d, 0xa8, 0x3c, 0xf4, 0x05, 0x10,
        0xb0, 0x0d, 0x6f, 0x1c, 0xa0, 0x17, 0x85, 0xae,
        0x68, 0xbf, 0xcc, 0x00, 0xc8, 0x86, 0x1b, 0x24,
        0x31, 0xc9, 0x49, 0x23, 0x91, 0xe0, 0x71, 0x29,
        0x06, 0x39, 0x39, 0x93, 0x49, 0x9c, 0x75, 0x18,
        0x1a, 0x8b, 0x61, 0x73, 0x1c, 0x7f, 0x37, 0xd5,
        0xf1, 0xab, 0x20, 0x5e, 0x62, 0x25, 0xeb, 0x58,
        0xd5, 0xfa, 0xc9, 0x7f, 0xad, 0x57, 0xd5, 0xcc,
        0x0d, 0xc1, 0x7a, 0x2b, 0x33, 0x2a, 0x76, 0x84,
        0x33, 0x26, 0x97, 0xcf, 0x47, 0x9d, 0x72, 0x2a,
        0xc9, 0x39, 0xde, 0xa8, 0x42, 0x27, 0x2d, 0xdc,
        0xee, 0x00, 0x60, 0xd2, 0x4f, 0x13, 0xe0, 0xde,
        0xd5, 0xc7, 0xf6, 0x7d, 0x8b, 0x2a, 0x43, 0x49,
        0x40, 0x99, 0xc2, 0x61, 0x84, 0x8e, 0x57, 0x09,
        0x7c, 0xcc, 0x19, 0x46, 0xbd, 0x4c, 0xd2, 0x7c,
        0x7d, 0x02, 0x4d, 0x88, 0xdf, 0x58, 0x24, 0x80,
        0xeb, 0x19, 0x3b, 0x2a, 0x13, 0x2b, 0x19, 0x85,
        0x3c, 0xd8, 0x31, 0x03, 0x00, 0xa4, 0xd4, 0x57,
        0x23, 0x2c, 0x24, 0x37, 0xb3, 0x62, 0xea, 0x35,
        0x29, 0xd0, 0x2c, 0xac, 0xfd, 0xbd, 0xdf, 0x3d,
        0xa6, 0xce, 0xfa, 0x0d, 0x5b, 0xb6, 0x15, 0x8b,
        0xe3, 0x58, 0xe9, 0xad, 0x99, 0x87, 0x29, 0x51,
        0x8d, 0x97, 0xd7, 0xa9, 0x55, 0xf0, 0x72, 0x6e,
        0x4e, 0x58, 0xcb, 0x2b, 0x4d, 0xbd, 0xd0, 0x48,
        0x7d, 0x14, 0x86, 0xdb, 0x3f, 0xa2, 0x5f, 0x6e,
        0x35, 0x4a, 0xe1, 0x70, 0xb1, 0x53, 0x72, 0xb7,
        0xbc, 0xe9, 0x3d, 0x1b, 0x33, 0xc0, 0x54, 0x6f,
        0x43, 0x55, 0x76, 0x85, 0x7f, 0x9b, 0xa5, 0xb3,
        0xc1, 0x1d, 0xd3, 0xfe, 0xe2, 0xd5, 0x96, 0x3d,
        0xdd, 0x92, 0x04, 0xb1, 0xad, 0x75, 0xdb, 0x13,
        0x4e, 0x49, 0xfc, 0x35, 0x34, 0xc5, 0xda, 0x13,
        0x98, 0xb8, 0x12, 0xbe, 0xda, 0x90, 0x55, 0x7c,
        0x11, 0x6c, 0xbe, 0x2b, 0x8c, 0x51, 0x29, 0x23,
        0xc1, 0x51, 0xbc, 0x0c, 0x1c, 0xe2, 0x20, 0xfc,
        0xfe, 0xf2, 0xaa, 0x71, 0x9b, 0x21, 0xdf, 0x25,
        0x1f, 0x68, 0x21, 0x7e, 0xe1, 0xc9, 0x87, 0xa0,
        0x20, 0xf6, 0x8d, 0x4f, 0x27, 0x8c, 0x3c, 0x0f,
        0x9d, 0xf4, 0x69, 0x25, 0xaa, 0x49, 0xab, 0x94,
        0x22, 0x5a, 0x92, 0x3a, 0xba, 0xb4, 0xc2, 0x8c,
        0x5a, 0xaa, 0x04, 0xbf, 0x46, 0xc5, 0xaa, 0x93,
        0xab, 0x0d, 0xe9, 0x54, 0x6c, 0x3a, 0x64, 0xa6,
        0xa2, 0x21, 0x66, 0xee, 0x1c, 0x10, 0x21, 0x84,
        0xf2, 0x9e, 0xcc, 0x57, 0xac, 0xc2, 0x25, 0x62,
        0xad, 0xbb, 0x59, 0xef, 0x25, 0x61, 0x6c, 0x81,
        0x38, 0x8a, 0xdc, 0x8c, 0xeb, 0x7b, 0x18, 0x1d,
        0xaf, 0xa9, 0xc5, 0x9a, 0xf4, 0x49, 0x26, 0x8a,
        0x25, 0xc4, 0x3e, 0x31, 0x95, 0x28, 0xef, 0xf7,
        0x72, 0xe9, 0xc5, 0xaa, 0x59, 0x72, 0x2b, 0x67,
        0x47, 0xe8, 0x6b, 0x51, 0x05, 0x24, 0xb8, 0x18,
        0xb3, 0x34, 0x0f, 0x8c, 0x2b, 0x80, 0xba, 0x61,
        0x1c, 0xbe, 0x9e, 0x9a, 0x7c, 0xe3, 0x60, 0x5e,
        0x49, 0x02, 0xff, 0x50, 0x8a, 0x64, 0x28, 0x64,
        0x46, 0x7b, 0x83, 0x14, 0x72, 0x6e, 0x59, 0x9b,
        0x56, 0x09, 0xb4, 0xf0, 0xde, 0x52, 0xc3, 0xf3,
        0x58, 0x17, 0x6a, 0xae, 0xb1, 0x0f, 0xf4, 0x39,
        0xcc, 0xd8, 0xce, 0x4d, 0xe1, 0x51, 0x17, 0x88,
        0xe4, 0x98, 0xd9, 0xd1, 0xa9, 0x55, 0xbc, 0xbf,
        0x7e, 0xc4, 0x51, 0x96, 0xdb, 0x44, 0x1d, 0xcd,
        0x8d, 0x74, 0xad, 0xa7, 0x8f, 0x87, 0x83, 0x75,
        0xfc, 0x36, 0xb7, 0xd2, 0xd4, 0x89, 0x16, 0x97,
        0xe4, 0xc6, 0x2a, 0xe9, 0x65, 0xc8, 0xca, 0x1c,
        0xbd, 0x86, 0xaf, 0x57, 0x80, 0xf7, 0xdd, 0x42,
        0xc0, 0x3b, 0x3f, 0x87, 0x51, 0x02, 0x2f, 0xf8,
        0xd8, 0x68, 0x0f, 0x3d, 0x95, 0x2d, 0xf1, 0x67,
        0x09, 0xa6, 0x5d, 0x0b, 0x7e, 0x01, 0xb4, 0xb2,
        0x32, 0x01, 0xa8, 0xd0, 0x58, 0x0d, 0xe6, 0xa2,
        0xd8, 0x4b, 0x22, 0x10, 0x7d, 0x11, 0xf3, 0xc2,
        0x4e, 0xb8, 0x43, 0x8e, 0x31, 0x79, 0x59, 0xe2,
        0xc4, 0x96, 0x29, 0x17, 0x40, 0x06, 0x0d, 0xdf,
        0xdf, 0xc3, 0x02, 0x30, 0x2a, 0xd1, 0x8e, 0xf2,
        0xee, 0x2d, 0xd2, 0x12, 0x63, 0x5a, 0x1d, 0x3c,
        0xba, 0x4a, 0xc4, 0x56, 0x90, 0xc6, 0x12, 0x0b,
        0xe0, 0x04, 0x3f, 0x35, 0x59, 0x8e, 0x40, 0x75,
        0xf4, 0x4c, 0x10, 0x61, 0xb9, 0x30, 0x89, 0x7c,
        0x8d, 0x0e, 0x25, 0xb7, 0x5a, 0x6b, 0x97, 0x05,
        0xc6, 0x37, 0x80, 0x6e, 0x94, 0x56, 0xa8, 0x5f,
        0x03, 0x94, 0x59, 0xc8, 0xc5, 0x3e, 0xdc, 0x23,
        0xe5, 0x68, 0x4f, 0xd7, 0xbb, 0x6d, 0x7e, 0xc1,
        0x8d, 0xf9, 0xcc, 0x3f, 0x38, 0xad, 0x77, 0xb3,
        0x18, 0x61, 0xed, 0x04, 0xc0, 0x71, 0xa7, 0x96,
        0xb1, 0xaf, 0x1d, 0x69, 0x78, 0xda, 0x6d, 0x89,
        0x8b, 0x50, 0x75, 0x99, 0x44, 0xb3, 0xb2, 0x75,
        0xd1, 0xc8, 0x14, 0x40, 0xa1, 0x0a, 0xbf, 0xc4,
        0x45, 0xc4, 0xee, 0x12, 0x90, 0x76, 0x26, 0x64,
        0xb7, 0x73, 0x2e, 0x0b, 0x0c, 0xfa, 0xc3, 0x55,
        0x29, 0x24, 0x1b, 0x7a, 0x00, 0x27, 0x07, 0x26,
        0x36, 0xf0, 0x38, 0x1a, 0xe3, 0xb7, 0xc4, 0x8d,
        0x1c, 0x9c, 0xa9, 0xc0, 0xc1, 0x45, 0x91, 0x9e,
        0x86, 0xdd, 0x82, 0x94, 0x45, 0xfa, 0xcd, 0x5a,
        0x19, 0x12, 0x7d, 0xef, 0xda, 0x17, 0xad, 0x21,
        0x17, 0x89, 0x8b, 0x45, 0xa7, 0xf5, 0xed, 0x51,
        0x9e, 0x58, 0x13, 0xdc, 0x84, 0xa4, 0xe6, 0x37
    };

    /* 768 byte shared secret */
    static const byte shared_secret[] = {
        0x9a, 0x9c, 0x1c, 0xb7, 0x73, 0x2f, 0xf2, 0x12,
        0xed, 0x59, 0x01, 0xbb, 0x75, 0xf7, 0xf5, 0xe4,
        0xa0, 0xa8, 0xbc, 0x3f, 0x3f, 0xb6, 0xf7, 0x74,
        0x6e, 0xc4, 0xba, 0x6d, 0x6c, 0x4d, 0x93, 0x31,
        0x2b, 0xa7, 0xa4, 0xb3, 0x47, 0x8f, 0x77, 0x04,
        0xb5, 0xa5, 0xab, 0xca, 0x6b, 0x5a, 0xe2, 0x86,
        0x02, 0x60, 0xca, 0xb4, 0xd7, 0x5e, 0xe0, 0x0f,
        0x73, 0xdd, 0xa2, 0x38, 0x7c, 0xae, 0x0f, 0x5a,
        0x1a, 0xd7, 0xfd, 0xb6, 0xc8, 0x6f, 0xdd, 0xe0,
        0x98, 0xd5, 0x07, 0xea, 0x1f, 0x2a, 0xbb, 0x9e,
        0xef, 0x01, 0x24, 0x04, 0xee, 0xf5, 0x89, 0xb1,
        0x12, 0x26, 0x54, 0x95, 0xef, 0xcb, 0x84, 0xe9,
        0xae, 0x05, 0xef, 0x63, 0x25, 0x15, 0x65, 0x79,
        0x79, 0x79, 0x91, 0xc3, 0x76, 0x72, 0xb4, 0x85,
        0x86, 0xd9, 0xd3, 0x03, 0xb0, 0xff, 0x04, 0x96,
        0x05, 0x3c, 0xde, 0xbf, 0x47, 0x34, 0x76, 0x70,
        0x17, 0xd2, 0x24, 0x83, 0xb9, 0xbb, 0xcf, 0x70,
        0x7c, 0xb8, 0xc6, 0x7b, 0x4e, 0x01, 0x86, 0x36,
        0xc7, 0xc5, 0xe5, 0x8b, 0x7c, 0x69, 0x74, 0x9a,
        0xfe, 0x1f, 0x58, 0x85, 0x0f, 0x00, 0xf8, 0x4e,
        0xf1, 0x56, 0xdc, 0xd1, 0x11, 0x28, 0x2c, 0xcf,
        0x6c, 0xb9, 0xc9, 0x57, 0x17, 0x2e, 0x19, 0x19,
        0x55, 0xb3, 0x4c, 0xd8, 0xfb, 0xe7, 0x6f, 0x70,
        0x63, 0xf9, 0x53, 0x45, 0xdd, 0xd5, 0x62, 0x95,
        0xd3, 0x7d, 0x7e, 0xa0, 0x00, 0x1a, 0x62, 0x9f,
        0x96, 0x0a, 0x5d, 0x0a, 0x25, 0x02, 0xbb, 0xff,
        0x5a, 0xe8, 0x9e, 0x5a, 0x66, 0x08, 0x93, 0xbc,
        0x92, 0xaf, 0xd2, 0x28, 0x04, 0x97, 0xc1, 0x54,
        0xfe, 0xcc, 0x0a, 0x25, 0xa2, 0xf4, 0x1d, 0x5a,
        0x9a, 0xb1, 0x3e, 0x9c, 0xba, 0x78, 0xe2, 0xcf,
        0x71, 0x70, 0xe3, 0x40, 0xea, 0xba, 0x69, 0x9b,
        0x03, 0xdd, 0x99, 0x26, 0x09, 0x84, 0x9d, 0x69,
        0x4d, 0x3d, 0x0b, 0xe9, 0x3f, 0x51, 0xcd, 0x05,
        0xe5, 0x00, 0xaf, 0x2c, 0xd3, 0xf6, 0xc0, 0x68,
        0xb5, 0x23, 0x53, 0x33, 0x14, 0xbd, 0x39, 0x1c,
        0xbd, 0x1b, 0xe6, 0x72, 0x90, 0xcc, 0xc2, 0x86,
        0x1a, 0x42, 0x83, 0x55, 0xb3, 0xed, 0x0b, 0x62,
        0x6d, 0x0e, 0xbb, 0x9e, 0x2a, 0x42, 0x32, 0x05,
        0x3f, 0xf2, 0x2c, 0xc8, 0x9f, 0x3c, 0xd2, 0xb1,
        0x0b, 0xb6, 0x4c, 0xa0, 0x22, 0x36, 0xee, 0xb9,
        0x55, 0x23, 0x3e, 0x80, 0xc7, 0x28, 0x7c, 0x39,
        0x11, 0xd3, 0x4a, 0x96, 0x2e, 0xef, 0x52, 0x34,
        0xf2, 0xda, 0xb1, 0xc6, 0xf5, 0x02, 0x10, 0xbf,
        0x56, 0x6b, 0x50, 0x56, 0xcd, 0x2c, 0xfe, 0xe1,
        0x94, 0x14, 0x19, 0x24, 0x6e, 0x9a, 0xdf, 0x0c,
        0xb8, 0xe2, 0xb8, 0xd5, 0xa3, 0xc1, 0x22, 0x8e,
        0x84, 0x92, 0x00, 0x16, 0xf1, 0x3f, 0x83, 0xf6,
        0x36, 0x31, 0xa5, 0x38, 0xc6, 0xcf, 0xf8, 0x9b,
        0x03, 0xc7, 0x6f, 0xb9, 0xa1, 0x04, 0xdf, 0x20,
        0x0f, 0x0b, 0x0f, 0x70, 0xff, 0x57, 0x36, 0x7f,
        0xb3, 0x6b, 0xcb, 0x8f, 0x48, 0xf7, 0xb2, 0xdb,
        0x85, 0x05, 0xd1, 0xfe, 0x34, 0x05, 0xf6, 0x57,
        0xb4, 0x5b, 0xcc, 0x3f, 0x0e, 0xba, 0x36, 0x59,
        0xb0, 0xfd, 0x4d, 0xf6, 0xf4, 0x5e, 0xd2, 0x65,
        0x1d, 0x98, 0x87, 0xb4, 0x5e, 0xff, 0x29, 0xaa,
        0x84, 0x9b, 0x44, 0x0f, 0x06, 0x36, 0x61, 0xbd,
        0xdb, 0x51, 0xda, 0x56, 0xc2, 0xd6, 0x19, 0xe2,
        0x57, 0x4f, 0xd0, 0x29, 0x71, 0xc8, 0xe4, 0xd6,
        0xfb, 0x8c, 0xd0, 0xfc, 0x4f, 0x25, 0x09, 0xa6,
        0xfc, 0x67, 0xe2, 0xb8, 0xac, 0xd3, 0x88, 0x8f,
        0x1f, 0xf6, 0xa1, 0xe3, 0x45, 0xa6, 0x34, 0xe3,
        0xb1, 0x6b, 0xb7, 0x37, 0x0e, 0x06, 0xc7, 0x63,
        0xde, 0xac, 0x3b, 0xac, 0x07, 0x91, 0x64, 0xcc,
        0x12, 0x10, 0x46, 0x85, 0x14, 0x0b, 0x6b, 0x03,
        0xba, 0x4a, 0x85, 0xae, 0xc5, 0x8c, 0xa5, 0x9d,
        0x36, 0x38, 0x33, 0xca, 0x42, 0x9c, 0x4b, 0x0c,
        0x46, 0xe1, 0x77, 0xe9, 0x1f, 0x80, 0xfe, 0xb7,
        0x1d, 0x5a, 0xf4, 0xc6, 0x11, 0x26, 0x78, 0xea,
        0x81, 0x25, 0x77, 0x47, 0xed, 0x8b, 0x59, 0xc2,
        0x6b, 0x49, 0xff, 0x83, 0x56, 0xec, 0xa5, 0xf0,
        0xe0, 0x8b, 0x15, 0xd4, 0x99, 0x40, 0x2a, 0x65,
        0x2a, 0x98, 0xf4, 0x71, 0x35, 0x63, 0x84, 0x08,
        0x4d, 0xcd, 0x71, 0x85, 0x55, 0xbc, 0xa4, 0x1c,
        0x90, 0x93, 0x03, 0x41, 0xde, 0xed, 0x78, 0x62,
        0x07, 0x30, 0x50, 0xac, 0x60, 0x21, 0x06, 0xc3,
        0xab, 0xa4, 0x04, 0xc0, 0xc2, 0x32, 0x07, 0xc4,
        0x1f, 0x2f, 0xec, 0xe2, 0x32, 0xbf, 0xbe, 0x5e,
        0x50, 0x5b, 0x2a, 0x19, 0x71, 0x44, 0x37, 0x76,
        0x8b, 0xbc, 0xdb, 0x73, 0x98, 0x65, 0x78, 0xc9,
        0x33, 0x97, 0x7e, 0xdc, 0x60, 0xa8, 0x87, 0xf2,
        0xb5, 0x96, 0x55, 0x7f, 0x44, 0x07, 0xcb, 0x3b,
        0xf3, 0xd7, 0x82, 0xfd, 0x77, 0x21, 0x82, 0x21,
        0x1a, 0x8b, 0xa2, 0xf5, 0x1f, 0x66, 0xd0, 0x57,
        0x00, 0x4f, 0xa9, 0xa5, 0x33, 0xb8, 0x69, 0x91,
        0xe8, 0x2e, 0xf7, 0x73, 0x47, 0x89, 0x30, 0x9b,
        0xb1, 0xfd, 0xe1, 0x5d, 0x11, 0xfd, 0x84, 0xd9,
        0xa2, 0x91, 0x1f, 0x8a, 0xa7, 0x7a, 0x77, 0x8e,
        0x3b, 0x10, 0x1d, 0x0a, 0x59, 0x50, 0x34, 0xb0,
        0xc3, 0x90, 0x9f, 0x56, 0xb7, 0x43, 0xeb, 0x51,
        0x99, 0x2b, 0x8e, 0x6d, 0x7b, 0x58, 0xe7, 0xc0,
        0x7f, 0x3d, 0xa0, 0x27, 0x50, 0xf2, 0x6e, 0xc8,
        0x1e, 0x7f, 0x84, 0xb3, 0xe1, 0xf7, 0x09, 0x85,
        0xd2, 0x9b, 0x56, 0x6b, 0xba, 0xa5, 0x19, 0x2e,
        0xec, 0xd8, 0x5c, 0xf5, 0x4e, 0x43, 0x36, 0x2e,
        0x89, 0x85, 0x41, 0x7f, 0x9c, 0x91, 0x2e, 0x62,
        0xc3, 0x41, 0xcf, 0x0e, 0xa1, 0x7f, 0xeb, 0x50
    };

    rc = linuxkm_test_kpp_driver(WOLFKM_FFDHE6144_DRIVER,
                                 b_pub, expected_a_pub, sizeof(b_pub),
                                 secret, sizeof(secret),
                                 shared_secret, sizeof(shared_secret));

    return rc;
}
#endif /* HAVE_FFDHE_6144 */

#ifdef HAVE_FFDHE_8192
static int linuxkm_test_ffdhe8192(void)
{
    int rc = 0;

    /* reference values from kernel crypto/testmgr.h */
    /* 1040 byte secret size */
    static const byte secret[] = {
#ifdef LITTLE_ENDIAN_ORDER
        0x01, 0x00, /* type */
        0x10, 0x04, /* len */
        0x00, 0x04, 0x00, 0x00, /* key_size */
        0x00, 0x00, 0x00, 0x00, /* p_size */
        0x00, 0x00, 0x00, 0x00, /* g_size */
#else
        0x00, 0x01, /* type */
        0x04, 0x10, /* len */
        0x00, 0x00, 0x04, 0x00, /* key_size */
        0x00, 0x00, 0x00, 0x00, /* p_size */
        0x00, 0x00, 0x00, 0x00, /* g_size */
#endif
        /* xa */
        0x76, 0x6e, 0xeb, 0xf9, 0xeb, 0x76, 0xae, 0x37,
        0xcb, 0x19, 0x49, 0x8b, 0xeb, 0xaf, 0xb0, 0x4b,
        0x6d, 0xe9, 0x15, 0xad, 0xda, 0xf2, 0xef, 0x58,
        0xe9, 0xd6, 0xdd, 0x4c, 0xb3, 0x56, 0xd0, 0x3b,
        0x00, 0xb0, 0x65, 0xed, 0xae, 0xe0, 0x2e, 0xdf,
        0x8f, 0x45, 0x3f, 0x3c, 0x5d, 0x2f, 0xfa, 0x96,
        0x36, 0x33, 0xb2, 0x01, 0x8b, 0x0f, 0xe8, 0x46,
        0x15, 0x6d, 0x60, 0x5b, 0xec, 0x32, 0xc3, 0x3b,
        0x06, 0xf3, 0xb4, 0x1b, 0x9a, 0xef, 0x3c, 0x03,
        0x0e, 0xcc, 0xce, 0x1d, 0x24, 0xa0, 0xc9, 0x08,
        0x65, 0xf9, 0x45, 0xe5, 0xd2, 0x43, 0x08, 0x88,
        0x58, 0xd6, 0x46, 0xe7, 0xbb, 0x25, 0xac, 0xed,
        0x3b, 0xac, 0x6f, 0x5e, 0xfb, 0xd6, 0x19, 0xa6,
        0x20, 0x3a, 0x1d, 0x0c, 0xe8, 0x00, 0x72, 0x54,
        0xd7, 0xd9, 0xc9, 0x26, 0x49, 0x18, 0xc6, 0xb8,
        0xbc, 0xdd, 0xf3, 0xce, 0xf3, 0x7b, 0x69, 0x04,
        0x5c, 0x6f, 0x11, 0xdb, 0x44, 0x42, 0x72, 0xb6,
        0xb7, 0x84, 0x17, 0x86, 0x47, 0x3f, 0xc5, 0xa1,
        0xd8, 0x86, 0xef, 0xe2, 0x27, 0x49, 0x2b, 0x8f,
        0x3e, 0x91, 0x12, 0xd9, 0x45, 0x96, 0xf7, 0xe6,
        0x77, 0x76, 0x36, 0x58, 0x71, 0x9a, 0xb1, 0xdb,
        0xcf, 0x24, 0x9e, 0x7e, 0xad, 0xce, 0x45, 0xba,
        0xb5, 0xec, 0x8e, 0xb9, 0xd6, 0x7b, 0x3d, 0x76,
        0xa4, 0x85, 0xad, 0xd8, 0x49, 0x9b, 0x80, 0x9d,
        0x7f, 0x9f, 0x85, 0x09, 0x9e, 0x86, 0x5b, 0x6b,
        0xf3, 0x8d, 0x39, 0x5e, 0x6f, 0xe4, 0x30, 0xc8,
        0xa5, 0xf3, 0xdf, 0x68, 0x73, 0x6b, 0x2e, 0x9a,
        0xcb, 0xac, 0x0a, 0x0d, 0x44, 0xc1, 0xaf, 0xb2,
        0x11, 0x1b, 0x7c, 0x43, 0x08, 0x44, 0x43, 0xe2,
        0x4e, 0xfd, 0x93, 0x30, 0x99, 0x09, 0x12, 0xbb,
        0xf6, 0x31, 0x34, 0xa5, 0x3d, 0x45, 0x98, 0xee,
        0xd7, 0x2a, 0x1a, 0x89, 0xf5, 0x37, 0x92, 0x33,
        0xa0, 0xdd, 0xf5, 0xfb, 0x1f, 0x90, 0x42, 0x55,
        0x5a, 0x0b, 0x82, 0xff, 0xf0, 0x96, 0x92, 0x15,
        0x65, 0x5a, 0x55, 0x96, 0xca, 0x1b, 0xd5, 0xe5,
        0xb5, 0x94, 0xde, 0x2e, 0xa6, 0x03, 0x57, 0x9e,
        0x15, 0xe4, 0x32, 0x2b, 0x1f, 0xb2, 0x22, 0x21,
        0xe9, 0xa0, 0x05, 0xd3, 0x65, 0x6c, 0x11, 0x66,
        0x25, 0x38, 0xbb, 0xa3, 0x6c, 0xc2, 0x0b, 0x2b,
        0xd0, 0x7a, 0x20, 0x26, 0x29, 0x37, 0x5d, 0x5f,
        0xd8, 0xff, 0x2a, 0xcd, 0x46, 0x6c, 0xd6, 0x6e,
        0xe5, 0x77, 0x1a, 0xe6, 0x33, 0xf1, 0x8e, 0xc8,
        0x10, 0x30, 0x11, 0x00, 0x27, 0xf9, 0x7d, 0x0e,
        0x28, 0x43, 0xa7, 0x67, 0x38, 0x7f, 0x16, 0xda,
        0xd0, 0x01, 0x8e, 0xa4, 0xe8, 0x6f, 0xcd, 0x23,
        0xaf, 0x77, 0x52, 0x34, 0xad, 0x7e, 0xc3, 0xed,
        0x2d, 0x10, 0x0a, 0x33, 0xdc, 0xcf, 0x1b, 0x88,
        0x0f, 0xcc, 0x48, 0x7f, 0x42, 0xf0, 0x9e, 0x13,
        0x1f, 0xf5, 0xd1, 0xe9, 0x90, 0x87, 0xbd, 0xfa,
        0x5f, 0x1d, 0x77, 0x55, 0xcb, 0xc3, 0x05, 0xaf,
        0x71, 0xd0, 0xe0, 0xab, 0x46, 0x31, 0xd7, 0xea,
        0x89, 0x54, 0x2d, 0x39, 0xaf, 0xf6, 0x4f, 0x74,
        0xaf, 0x46, 0x58, 0x89, 0x78, 0x95, 0x2e, 0xe6,
        0x90, 0xb7, 0xaa, 0x00, 0x73, 0x9f, 0xed, 0xb9,
        0x00, 0xd6, 0xf6, 0x6d, 0x26, 0x59, 0xcd, 0x56,
        0xdb, 0xf7, 0x3d, 0x5f, 0xeb, 0x6e, 0x46, 0x33,
        0xb1, 0x23, 0xed, 0x9f, 0x8d, 0x58, 0xdc, 0xb4,
        0x28, 0x3b, 0x90, 0x09, 0xc4, 0x61, 0x02, 0x1f,
        0xf8, 0x62, 0xf2, 0x6e, 0xc1, 0x94, 0x71, 0x66,
        0x93, 0x11, 0xdf, 0xaa, 0x3e, 0xd7, 0xb5, 0xe5,
        0xc1, 0x78, 0xe9, 0x14, 0xcd, 0x55, 0x16, 0x51,
        0xdf, 0x8d, 0xd0, 0x94, 0x8c, 0x43, 0xe9, 0xb8,
        0x1d, 0x42, 0x7f, 0x76, 0xbc, 0x6f, 0x87, 0x42,
        0x88, 0xde, 0xd7, 0x52, 0x78, 0x00, 0x4f, 0x18,
        0x02, 0xe7, 0x7b, 0xe2, 0x8a, 0xc3, 0xd1, 0x43,
        0xa5, 0xac, 0xda, 0xb0, 0x8d, 0x19, 0x96, 0xd4,
        0x81, 0xe0, 0x75, 0xe9, 0xca, 0x41, 0x7e, 0x1f,
        0x93, 0x0b, 0x26, 0x24, 0xb3, 0xaa, 0xdd, 0x10,
        0x20, 0xd3, 0xf2, 0x9f, 0x3f, 0xdf, 0x65, 0xde,
        0x67, 0x79, 0xdc, 0x76, 0x9f, 0x3c, 0x72, 0x75,
        0x65, 0x8a, 0x30, 0xcc, 0xd2, 0xcc, 0x06, 0xb1,
        0xab, 0x62, 0x86, 0x78, 0x5d, 0xb8, 0xce, 0x72,
        0xb3, 0x12, 0xc7, 0x9f, 0x07, 0xd0, 0x6b, 0x98,
        0x82, 0x9b, 0x6c, 0xbb, 0x15, 0xe5, 0xcc, 0xf4,
        0xc8, 0xf4, 0x60, 0x81, 0xdc, 0xd3, 0x09, 0x1b,
        0x5e, 0xd4, 0xf3, 0x55, 0xcf, 0x1c, 0x16, 0x83,
        0x61, 0xb4, 0x2e, 0xcc, 0x08, 0x67, 0x58, 0xfd,
        0x46, 0x64, 0xbc, 0x29, 0x4b, 0xdd, 0xda, 0xec,
        0xdc, 0xc6, 0xa9, 0xa5, 0x73, 0xfb, 0xf8, 0xf3,
        0xaf, 0x89, 0xa8, 0x9e, 0x25, 0x14, 0xfa, 0xac,
        0xeb, 0x1c, 0x7c, 0x80, 0x96, 0x66, 0x4d, 0x41,
        0x67, 0x9b, 0x07, 0x4f, 0x0a, 0x97, 0x17, 0x1c,
        0x4d, 0x61, 0xc7, 0x2e, 0x6f, 0x36, 0x98, 0x29,
        0x50, 0x39, 0x6d, 0xe7, 0x70, 0xda, 0xf0, 0xc8,
        0x05, 0x80, 0x7b, 0x32, 0xff, 0xfd, 0x12, 0xde,
        0x61, 0x0d, 0xf9, 0x4c, 0x21, 0xf1, 0x56, 0x72,
        0x3d, 0x61, 0x46, 0xc0, 0x2d, 0x07, 0xd1, 0x6c,
        0xd3, 0xbe, 0x9a, 0x21, 0x83, 0x85, 0xf7, 0xed,
        0x53, 0x95, 0x44, 0x40, 0x8f, 0x75, 0x12, 0x18,
        0xc2, 0x9a, 0xfd, 0x5e, 0xce, 0x66, 0xa6, 0x7f,
        0x57, 0xc0, 0xd7, 0x73, 0x76, 0xb3, 0x13, 0xda,
        0x2e, 0x58, 0xc6, 0x27, 0x40, 0xb2, 0x2d, 0xef,
        0x7d, 0x72, 0xb4, 0xa8, 0x75, 0x6f, 0xcc, 0x5f,
        0x42, 0x3e, 0x2c, 0x90, 0x36, 0x59, 0xa0, 0x34,
        0xaa, 0xce, 0xbc, 0x04, 0x4c, 0xe6, 0x56, 0xc2,
        0xcd, 0xa6, 0x1c, 0x59, 0x04, 0x56, 0x53, 0xcf,
        0x6d, 0xd7, 0xf0, 0xb1, 0x4f, 0x91, 0xfa, 0x84,
        0xcf, 0x4b, 0x8d, 0x50, 0x4c, 0xf8, 0x2a, 0x31,
        0x5f, 0xe3, 0xba, 0x79, 0xb4, 0xcc, 0x59, 0x64,
        0xe3, 0x7a, 0xfa, 0xf6, 0x06, 0x9d, 0x04, 0xbb,
        0xce, 0x61, 0xbf, 0x9e, 0x59, 0x0a, 0x09, 0x51,
        0x6a, 0xbb, 0x0b, 0x80, 0xe0, 0x91, 0xc1, 0x51,
        0x04, 0x58, 0x67, 0x67, 0x4b, 0x42, 0x4f, 0x95,
        0x68, 0x75, 0xe2, 0x1f, 0x9c, 0x14, 0x70, 0xfd,
        0x3a, 0x8a, 0xce, 0x8b, 0x04, 0xa1, 0x89, 0xe7,
        0xb4, 0xbf, 0x70, 0xfe, 0xf3, 0x0c, 0x48, 0x04,
        0x3a, 0xd2, 0x85, 0x68, 0x03, 0xe7, 0xfa, 0xec,
        0x5b, 0x55, 0xb7, 0x95, 0xfd, 0x5b, 0x19, 0x35,
        0xad, 0xcb, 0x4a, 0x63, 0x03, 0x44, 0x64, 0x2a,
        0x48, 0x59, 0x9a, 0x26, 0x43, 0x96, 0x8c, 0xe6,
        0xbd, 0xb7, 0x90, 0xd4, 0x5f, 0x8d, 0x08, 0x28,
        0xa8, 0xc5, 0x89, 0x70, 0xb9, 0x6e, 0xd3, 0x3b,
        0x76, 0x0e, 0x37, 0x98, 0x15, 0x27, 0xca, 0xc9,
        0xb0, 0xe0, 0xfd, 0xf3, 0xc6, 0xdf, 0x69, 0xce,
        0xe1, 0x5f, 0x6a, 0x3e, 0x5c, 0x86, 0xe2, 0x58,
        0x41, 0x11, 0xf0, 0x7e, 0x56, 0xec, 0xe4, 0xc9,
        0x0d, 0x87, 0x91, 0xfb, 0xb9, 0xc8, 0x0d, 0x34,
        0xab, 0xb0, 0xc6, 0xf2, 0xa6, 0x00, 0x7b, 0x18,
        0x92, 0xf4, 0x43, 0x7f, 0x01, 0x85, 0x2e, 0xef,
        0x8c, 0x72, 0x50, 0x10, 0xdb, 0xf1, 0x37, 0x62,
        0x16, 0x85, 0x71, 0x01, 0xa8, 0x2b, 0xf0, 0x13,
        0xd3, 0x7c, 0x0b, 0xaf, 0xf1, 0xf3, 0xd1, 0xee,
        0x90, 0x41, 0x5f, 0x7d, 0x5b, 0xa9, 0x83, 0x4b,
        0xfa, 0x80, 0x59, 0x50, 0x73, 0xe1, 0xc4, 0xf9,
        0x5e, 0x4b, 0xde, 0xd9, 0xf5, 0x22, 0x68, 0x5e,
        0x65, 0xd9, 0x37, 0xe4, 0x1a, 0x08, 0x0e, 0xb1,
        0x28, 0x2f, 0x40, 0x9e, 0x37, 0xa8, 0x12, 0x56,
        0xb7, 0xb8, 0x64, 0x94, 0x68, 0x94, 0xff, 0x9f
    };

    /* 1024 byte pub key */
    static const byte b_pub[] = {
        0x26, 0xa8, 0x3a, 0x97, 0xe0, 0x52, 0x76, 0x07,
        0x26, 0xa7, 0xbb, 0x21, 0xfd, 0xe5, 0x69, 0xde,
        0xe6, 0xe0, 0xb5, 0xa0, 0xf1, 0xaa, 0x51, 0x2b,
        0x56, 0x1c, 0x3c, 0x6c, 0xe5, 0x9f, 0x8f, 0x75,
        0x71, 0x04, 0x86, 0xf6, 0x43, 0x2f, 0x20, 0x7f,
        0x45, 0x4f, 0x5c, 0xb9, 0xf3, 0x90, 0xbe, 0xa9,
        0xa0, 0xd7, 0xe8, 0x03, 0x0e, 0xfe, 0x99, 0x9b,
        0x8a, 0x1c, 0xbe, 0xa7, 0x63, 0xe8, 0x2b, 0x45,
        0xd4, 0x2c, 0x65, 0x25, 0x4c, 0x33, 0xda, 0xc5,
        0x85, 0x77, 0x5d, 0x62, 0xea, 0x93, 0xe4, 0x45,
        0x59, 0xff, 0xa1, 0xd2, 0xf1, 0x73, 0x11, 0xed,
        0x02, 0x64, 0x8a, 0x1a, 0xfb, 0xe1, 0x88, 0xa6,
        0x50, 0x6f, 0xff, 0x87, 0x12, 0xbb, 0xfc, 0x10,
        0xcf, 0x19, 0x41, 0xb0, 0x35, 0x44, 0x7d, 0x51,
        0xe9, 0xc0, 0x77, 0xf2, 0x73, 0x21, 0x2e, 0x62,
        0xbf, 0x65, 0xa5, 0xd1, 0x3b, 0xb1, 0x3e, 0x19,
        0x75, 0x4b, 0xb7, 0x8e, 0x03, 0xc3, 0xdf, 0xc8,
        0xb2, 0xe6, 0xec, 0x2d, 0x7d, 0xa5, 0x6a, 0xba,
        0x93, 0x47, 0x50, 0xeb, 0x6e, 0xdb, 0x88, 0x05,
        0x45, 0xad, 0x03, 0x8c, 0xf7, 0x9a, 0xe1, 0xc9,
        0x1e, 0x16, 0x96, 0x37, 0xa5, 0x3e, 0xe9, 0xb9,
        0xa8, 0xdc, 0xb9, 0xa9, 0xf6, 0xa1, 0x3d, 0xed,
        0xbe, 0x12, 0x29, 0x8a, 0x3d, 0x3d, 0x90, 0xfc,
        0x94, 0xfe, 0x66, 0x28, 0x1c, 0x1b, 0xa4, 0x89,
        0x47, 0x66, 0x4f, 0xac, 0x14, 0x00, 0x22, 0x2d,
        0x5c, 0x03, 0xea, 0x71, 0x4d, 0x19, 0x7d, 0xd6,
        0x58, 0x39, 0x4c, 0x3d, 0x06, 0x2b, 0x30, 0xa6,
        0xdc, 0x2c, 0x8d, 0xd1, 0xde, 0x79, 0x77, 0xfa,
        0x9c, 0x6b, 0x72, 0x11, 0x8a, 0x7f, 0x7d, 0x37,
        0x28, 0x2a, 0x88, 0xbf, 0x0a, 0xdb, 0xac, 0x3b,
        0xc5, 0xa5, 0xd5, 0x7e, 0x25, 0xec, 0xa6, 0x7f,
        0x5b, 0x53, 0x75, 0x83, 0x49, 0xd4, 0x77, 0xcc,
        0x7d, 0x7e, 0xd3, 0x3d, 0x30, 0x2c, 0x98, 0x3f,
        0x18, 0x9a, 0x11, 0x8a, 0x37, 0xda, 0x99, 0x0f,
        0x3b, 0x06, 0xe1, 0x87, 0xd5, 0xe9, 0x4e, 0xe0,
        0x9c, 0x0e, 0x39, 0x34, 0xe2, 0xdd, 0xf6, 0x58,
        0x60, 0x63, 0xa6, 0xea, 0xe8, 0xc0, 0xb4, 0xde,
        0xdf, 0xa0, 0xbc, 0x21, 0xc3, 0x2d, 0xf4, 0xa4,
        0xc8, 0x6f, 0x62, 0x6c, 0x0f, 0x71, 0x88, 0xf9,
        0xda, 0x2d, 0x30, 0xd5, 0x95, 0xe1, 0xfc, 0x6d,
        0x88, 0xc5, 0xc3, 0x95, 0x51, 0x83, 0xde, 0x41,
        0x46, 0x6f, 0x7e, 0x1b, 0x10, 0x48, 0xad, 0x2b,
        0x82, 0x88, 0xa2, 0x6f, 0x57, 0x4d, 0x4a, 0xbd,
        0x90, 0xc8, 0x06, 0x8f, 0x52, 0x5d, 0x6e, 0xee,
        0x09, 0xe6, 0xa3, 0xcb, 0x30, 0x9c, 0x14, 0xf6,
        0xac, 0x66, 0x9b, 0x81, 0x0a, 0x75, 0x42, 0x6b,
        0xab, 0x27, 0xec, 0x76, 0xfb, 0x8d, 0xc5, 0xbf,
        0x0e, 0x93, 0x81, 0x7b, 0x81, 0xd4, 0x85, 0xa6,
        0x90, 0x5a, 0xa6, 0xa2, 0x8b, 0xa9, 0xb7, 0x34,
        0xe6, 0x15, 0x36, 0x93, 0x8b, 0xe2, 0x99, 0xc7,
        0xad, 0x66, 0x7e, 0xd6, 0x89, 0xa9, 0xc8, 0x15,
        0xcb, 0xc5, 0xeb, 0x06, 0x85, 0xd4, 0x2f, 0x6e,
        0x9b, 0x95, 0x7a, 0x06, 0x6c, 0xfa, 0x31, 0x1d,
        0xc4, 0xe5, 0x7d, 0xfb, 0x10, 0x35, 0x88, 0xc2,
        0xbe, 0x1c, 0x16, 0x5d, 0xc2, 0xf4, 0x0d, 0xf3,
        0xc9, 0x94, 0xb2, 0x7e, 0xa7, 0xbd, 0x9c, 0x03,
        0x32, 0xaf, 0x8b, 0x1a, 0xc8, 0xcc, 0x82, 0xd8,
        0x87, 0x96, 0x6e, 0x3d, 0xcc, 0x93, 0xd2, 0x43,
        0x73, 0xf9, 0xde, 0xec, 0x49, 0x49, 0xf4, 0x56,
        0x2a, 0xc8, 0x6e, 0x32, 0x70, 0x48, 0xf8, 0x70,
        0xa3, 0x96, 0x31, 0xf4, 0xf2, 0x08, 0xc5, 0x12,
        0xd2, 0xeb, 0xb6, 0xea, 0xa3, 0x07, 0x05, 0x61,
        0x74, 0xa3, 0x04, 0x2f, 0x17, 0x82, 0x40, 0x5e,
        0x4c, 0xd1, 0x51, 0xb8, 0x10, 0x5b, 0xc8, 0x9f,
        0x87, 0x73, 0x80, 0x0d, 0x6f, 0xc6, 0xb9, 0xf6,
        0x7c, 0x31, 0x0a, 0xcc, 0xd9, 0x03, 0x0f, 0x7a,
        0x47, 0x69, 0xb1, 0x55, 0xab, 0xe9, 0xb5, 0x75,
        0x62, 0x9e, 0x95, 0xbe, 0x7b, 0xa9, 0x53, 0x6e,
        0x28, 0x73, 0xdc, 0xb3, 0xa4, 0x8a, 0x1c, 0x91,
        0xf5, 0x8a, 0xf9, 0x32, 0x2b, 0xbd, 0xa5, 0xdc,
        0x07, 0xb5, 0xaf, 0x49, 0xdb, 0x9c, 0x35, 0xc9,
        0x69, 0xde, 0xac, 0xb1, 0xd0, 0x86, 0xcb, 0x31,
        0x0b, 0xc4, 0x4f, 0x63, 0x4e, 0x70, 0xa7, 0x80,
        0xe3, 0xbc, 0x0b, 0x73, 0x0e, 0xf2, 0x8c, 0x87,
        0x88, 0x7b, 0xa9, 0x6d, 0xde, 0x8a, 0x73, 0x14,
        0xb9, 0x80, 0x55, 0x03, 0x2b, 0x29, 0x64, 0x6a,
        0xda, 0x48, 0x0e, 0x78, 0x07, 0x40, 0x48, 0x46,
        0x58, 0xa9, 0x4e, 0x68, 0x1d, 0xd1, 0xc1, 0xc8,
        0x3b, 0x35, 0x53, 0x61, 0xd5, 0xe3, 0x0d, 0x4c,
        0x42, 0x74, 0x10, 0x67, 0x85, 0x9f, 0x66, 0x2a,
        0xf7, 0x2b, 0x7b, 0x77, 0x8b, 0x6e, 0xda, 0x2c,
        0xc1, 0x5a, 0x20, 0x34, 0x3f, 0xf5, 0x8b, 0x6f,
        0xe4, 0x61, 0xf5, 0x58, 0xab, 0x72, 0x1a, 0xf1,
        0x8d, 0x28, 0xcc, 0xa5, 0x30, 0x68, 0xb5, 0x50,
        0x7b, 0x81, 0x43, 0x89, 0x8e, 0xa9, 0xac, 0x63,
        0x3a, 0x4a, 0x78, 0x7b, 0xd2, 0x45, 0xe6, 0xe0,
        0xdc, 0x5d, 0xf2, 0x1a, 0x2b, 0x54, 0x50, 0xa5,
        0x9d, 0xf6, 0xe7, 0x9f, 0x25, 0xaf, 0x56, 0x6a,
        0x84, 0x2a, 0x75, 0xa3, 0x9a, 0xc7, 0xfa, 0x94,
        0xec, 0x83, 0xab, 0xa5, 0xaa, 0xe1, 0xf9, 0x89,
        0x29, 0xa9, 0xf6, 0x53, 0x24, 0x24, 0xae, 0x4a,
        0xe8, 0xbc, 0xe8, 0x9e, 0x5c, 0xd7, 0x54, 0x7c,
        0x65, 0x20, 0x97, 0x28, 0x94, 0x76, 0xf9, 0x9e,
        0x81, 0xcf, 0x98, 0x6a, 0x3a, 0x7b, 0xec, 0xf3,
        0x09, 0x60, 0x2e, 0x43, 0x18, 0xb5, 0xf6, 0x8c,
        0x44, 0x0f, 0xf2, 0x0a, 0x17, 0x5b, 0xac, 0x98,
        0x30, 0xab, 0x6e, 0xd5, 0xb3, 0xef, 0x25, 0x68,
        0x50, 0xb6, 0xe1, 0xc0, 0xe4, 0x5a, 0x63, 0x43,
        0xea, 0xca, 0xda, 0x23, 0xc1, 0xc2, 0xe9, 0x30,
        0xec, 0xb3, 0x9f, 0xbf, 0x1f, 0x09, 0x76, 0xaf,
        0x65, 0xbc, 0xb5, 0xab, 0x30, 0xac, 0x0b, 0x05,
        0xef, 0x5c, 0xa3, 0x65, 0x77, 0x33, 0x1c, 0xc5,
        0xdf, 0xc9, 0x39, 0xab, 0xca, 0xf4, 0x3b, 0x88,
        0x25, 0x6d, 0x50, 0x87, 0xb1, 0x79, 0xc2, 0x23,
        0x9d, 0xb5, 0x21, 0x01, 0xaa, 0xa3, 0xb7, 0x61,
        0xa3, 0x48, 0x91, 0x72, 0x3d, 0x54, 0x85, 0x86,
        0x91, 0x81, 0x35, 0x78, 0xbf, 0x8f, 0x27, 0x57,
        0xcb, 0x9b, 0x34, 0xab, 0x63, 0x40, 0xf1, 0xbc,
        0x23, 0x5a, 0x26, 0x6a, 0xba, 0x57, 0xe2, 0x8f,
        0x2a, 0xdc, 0x82, 0xe0, 0x3b, 0x7f, 0xec, 0xd3,
        0xd8, 0x9d, 0xd3, 0x13, 0x54, 0x70, 0x64, 0xc3,
        0xfd, 0xbf, 0xa3, 0x46, 0xa7, 0x53, 0x42, 0x7f,
        0xc1, 0xbd, 0x7b, 0xb3, 0x13, 0x47, 0x2a, 0x45,
        0x1e, 0x76, 0x2c, 0x0d, 0x6d, 0x46, 0x26, 0x24,
        0xa8, 0xc7, 0x00, 0x2b, 0x10, 0x7f, 0x2a, 0x6c,
        0xfc, 0x68, 0x4e, 0x6e, 0x85, 0x53, 0x00, 0xaf,
        0xd5, 0xfb, 0x59, 0x64, 0xc7, 0x9b, 0x24, 0xd1,
        0x05, 0xdc, 0x34, 0x53, 0x6d, 0x27, 0xa9, 0x79,
        0xff, 0xd7, 0x5e, 0x7a, 0x40, 0x81, 0x8e, 0xc3,
        0xf2, 0x38, 0xc9, 0x8d, 0x87, 0xb5, 0x38, 0xda,
        0x43, 0x64, 0x1b, 0x59, 0x62, 0x88, 0xc1, 0x6e,
        0x85, 0x84, 0x33, 0xcd, 0x6d, 0x7b, 0x62, 0x1d,
        0x60, 0xf9, 0x98, 0xf7, 0xd1, 0xb1, 0xd4, 0xbe,
        0x56, 0x6e, 0xa8, 0x6f, 0xff, 0xe7, 0x8b, 0x60,
        0x53, 0x80, 0xc7, 0x7c, 0xe0, 0x78, 0x89, 0xa9,
        0xab, 0x42, 0x8f, 0x8e, 0x4d, 0x92, 0xac, 0xa7,
        0xfd, 0x47, 0x11, 0xc7, 0xdb, 0x7c, 0x77, 0xfb,
        0xa4, 0x1d, 0x70, 0xaf, 0x56, 0x14, 0x52, 0xb0
    };

    /* 1024 byte pub key */
    static const byte expected_a_pub[] = {
        0xa1, 0x6c, 0x9e, 0xda, 0x45, 0x4d, 0xf6, 0x59,
        0x04, 0x00, 0xc1, 0xc6, 0x8b, 0x12, 0x3b, 0xcd,
        0x07, 0xe4, 0x3e, 0xec, 0xac, 0x9b, 0xfc, 0xf7,
        0x6d, 0x73, 0x39, 0x9e, 0x52, 0xf8, 0xbe, 0x33,
        0xe2, 0xca, 0xea, 0x99, 0x76, 0xc7, 0xc9, 0x94,
        0x5c, 0xf3, 0x1b, 0xea, 0x6b, 0x66, 0x4b, 0x51,
        0x90, 0xf6, 0x4f, 0x75, 0xd5, 0x85, 0xf4, 0x28,
        0xfd, 0x74, 0xa5, 0x57, 0xb1, 0x71, 0x0c, 0xb6,
        0xb6, 0x95, 0x70, 0x2d, 0xfa, 0x4b, 0x56, 0xe0,
        0x56, 0x10, 0x21, 0xe5, 0x60, 0xa6, 0x18, 0xa4,
        0x78, 0x8c, 0x07, 0xc0, 0x2b, 0x59, 0x9c, 0x84,
        0x5b, 0xe9, 0xb9, 0x74, 0xbf, 0xbc, 0x65, 0x48,
        0x27, 0x82, 0x40, 0x53, 0x46, 0x32, 0xa2, 0x92,
        0x91, 0x9d, 0xf6, 0xd1, 0x07, 0x0e, 0x1d, 0x07,
        0x1b, 0x41, 0x04, 0xb1, 0xd4, 0xce, 0xae, 0x6e,
        0x46, 0xf1, 0x72, 0x50, 0x7f, 0xff, 0xa8, 0xa2,
        0xbc, 0x3a, 0xc1, 0xbb, 0x28, 0xd7, 0x7d, 0xcd,
        0x7a, 0x22, 0x01, 0xaf, 0x57, 0xb0, 0xa9, 0x02,
        0xd4, 0x8a, 0x92, 0xd5, 0xe6, 0x8e, 0x6f, 0x11,
        0x39, 0xfe, 0x36, 0x87, 0x89, 0x42, 0x25, 0x42,
        0xd9, 0xbe, 0x67, 0x15, 0xe1, 0x82, 0x8a, 0x5e,
        0x98, 0xc2, 0xd5, 0xde, 0x9e, 0x13, 0x1a, 0xe7,
        0xf9, 0x9f, 0x8e, 0x2d, 0x49, 0xdc, 0x4d, 0x98,
        0x8c, 0xdd, 0xfd, 0x24, 0x7c, 0x46, 0xa9, 0x69,
        0x3b, 0x31, 0xb3, 0x12, 0xce, 0x54, 0xf6, 0x65,
        0x75, 0x40, 0xc2, 0xf1, 0x04, 0x92, 0xe3, 0x83,
        0xeb, 0x02, 0x3d, 0x79, 0xc0, 0xf9, 0x7c, 0x28,
        0xb3, 0x97, 0x03, 0xf7, 0x61, 0x1c, 0xce, 0x95,
        0x1a, 0xa0, 0xb3, 0x77, 0x1b, 0xc1, 0x9f, 0xf8,
        0xf6, 0x3f, 0x4d, 0x0a, 0xfb, 0xfa, 0x64, 0x1c,
        0xcb, 0x37, 0x5b, 0xc3, 0x28, 0x60, 0x9f, 0xd1,
        0xf2, 0xc4, 0xee, 0x77, 0xaa, 0x1f, 0xe9, 0xa2,
        0x89, 0x4c, 0xc6, 0xb7, 0xb3, 0xe4, 0xa5, 0xed,
        0xa7, 0xe8, 0xac, 0x90, 0xdc, 0xc3, 0xfb, 0x56,
        0x9c, 0xda, 0x2c, 0x1d, 0x1a, 0x9a, 0x8c, 0x82,
        0x92, 0xee, 0xdc, 0xa0, 0xa4, 0x01, 0x6e, 0x7f,
        0xc7, 0x0e, 0xc2, 0x73, 0x7d, 0xa6, 0xac, 0x12,
        0x01, 0xc0, 0xc0, 0xc8, 0x7c, 0x84, 0x86, 0xc7,
        0xa5, 0x94, 0xe5, 0x33, 0x84, 0x71, 0x6e, 0x36,
        0xe3, 0x3b, 0x81, 0x30, 0xe0, 0xc8, 0x51, 0x52,
        0x2b, 0x9e, 0x68, 0xa2, 0x6e, 0x09, 0x95, 0x8c,
        0x7f, 0x78, 0x82, 0xbd, 0x53, 0x26, 0xe7, 0x95,
        0xe0, 0x03, 0xda, 0xc0, 0xc3, 0x6e, 0xcf, 0xdc,
        0xb3, 0x14, 0xfc, 0xe9, 0x5b, 0x9b, 0x70, 0x6c,
        0x93, 0x04, 0xab, 0x13, 0xf7, 0x17, 0x6d, 0xee,
        0xad, 0x32, 0x48, 0xe9, 0xa0, 0x94, 0x1b, 0x14,
        0x64, 0x4f, 0xa1, 0xb3, 0x8d, 0x6a, 0xca, 0x28,
        0xfe, 0x4a, 0xf4, 0xf0, 0xc5, 0xb7, 0xf9, 0x8a,
        0x8e, 0xff, 0xfe, 0x57, 0x6f, 0x20, 0xdb, 0x04,
        0xab, 0x02, 0x31, 0x22, 0x42, 0xfd, 0xbd, 0x77,
        0xea, 0xce, 0xe8, 0xc7, 0x5d, 0xe0, 0x8e, 0xd6,
        0x66, 0xd0, 0xe4, 0x04, 0x2f, 0x5f, 0x71, 0xc7,
        0x61, 0x2d, 0xa5, 0x3f, 0x2f, 0x46, 0xf2, 0xd8,
        0x5b, 0x25, 0x82, 0xf0, 0x52, 0x88, 0xc0, 0x59,
        0xd3, 0xa3, 0x90, 0x17, 0xc2, 0x04, 0x13, 0xc3,
        0x13, 0x69, 0x4f, 0x17, 0xb1, 0xb3, 0x46, 0x4f,
        0xa7, 0xe6, 0x8b, 0x5e, 0x3e, 0x95, 0x0e, 0xf5,
        0x42, 0x17, 0x7f, 0x4d, 0x1f, 0x1b, 0x7d, 0x65,
        0x86, 0xc5, 0xc8, 0xae, 0xae, 0xd8, 0x4f, 0xe7,
        0x89, 0x41, 0x69, 0xfd, 0x06, 0xce, 0x5d, 0xed,
        0x44, 0x55, 0xad, 0x51, 0x98, 0x15, 0x78, 0x8d,
        0x68, 0xfc, 0x93, 0x72, 0x9d, 0x22, 0xe5, 0x1d,
        0x21, 0xc3, 0xbe, 0x3a, 0x44, 0x34, 0xc0, 0xa3,
        0x1f, 0xca, 0xdf, 0x45, 0xd0, 0x5c, 0xcd, 0xb7,
        0x72, 0xeb, 0xae, 0x7a, 0xad, 0x3f, 0x05, 0xa0,
        0xe3, 0x6e, 0x5a, 0xd8, 0x52, 0xa7, 0xf1, 0x1e,
        0xb4, 0xf2, 0xcf, 0xe7, 0xdf, 0xa7, 0xf2, 0x22,
        0x00, 0xb2, 0xc4, 0x17, 0x3d, 0x2c, 0x15, 0x04,
        0x71, 0x28, 0x69, 0x5c, 0x69, 0x21, 0xc8, 0xf1,
        0x9b, 0xd8, 0xc7, 0xbc, 0x27, 0xa3, 0x85, 0xe9,
        0x53, 0x77, 0xd3, 0x65, 0xc3, 0x86, 0xdd, 0xb3,
        0x76, 0x13, 0xfb, 0xa1, 0xd4, 0xee, 0x9d, 0xe4,
        0x51, 0x3f, 0x83, 0x59, 0xe4, 0x47, 0xa8, 0xa6,
        0x0d, 0x68, 0xd5, 0xf6, 0xf4, 0xca, 0x31, 0xcd,
        0x30, 0x48, 0x34, 0x90, 0x11, 0x8e, 0x87, 0xe9,
        0xea, 0xc9, 0xd0, 0xc3, 0xba, 0x28, 0xf9, 0xc0,
        0xc9, 0x8e, 0x23, 0xe5, 0xc2, 0xee, 0xf2, 0x47,
        0x9c, 0x41, 0x1c, 0x10, 0x33, 0x27, 0x23, 0x49,
        0xe5, 0x0d, 0x18, 0xbe, 0x19, 0xc1, 0xba, 0x6c,
        0xdc, 0xb7, 0xa1, 0xe7, 0xc5, 0x0d, 0x6f, 0xf0,
        0x8c, 0x62, 0x6e, 0x0d, 0x14, 0xef, 0xef, 0xf2,
        0x8e, 0x01, 0xd2, 0x76, 0xf5, 0xc1, 0xe1, 0x92,
        0x3c, 0xb3, 0x76, 0xcd, 0xd8, 0xdd, 0x9b, 0xe0,
        0x8e, 0xdc, 0x24, 0x34, 0x13, 0x65, 0x0f, 0x11,
        0xaf, 0x99, 0x7a, 0x2f, 0xe6, 0x1f, 0x7d, 0x17,
        0x3e, 0x8a, 0x68, 0x9a, 0x37, 0xc8, 0x8d, 0x3e,
        0xa3, 0xfe, 0xfe, 0x57, 0x22, 0xe6, 0x0e, 0x50,
        0xb5, 0x98, 0x0b, 0x71, 0xd8, 0x01, 0xa2, 0x8d,
        0x51, 0x96, 0x50, 0xc2, 0x41, 0x31, 0xd8, 0x23,
        0x98, 0xfc, 0xd1, 0x9d, 0x7e, 0x27, 0xbb, 0x69,
        0x78, 0xe0, 0x87, 0xf7, 0xe4, 0xdd, 0x58, 0x13,
        0x9d, 0xec, 0x00, 0xe4, 0xb9, 0x70, 0xa2, 0x94,
        0x5d, 0x52, 0x4e, 0xf2, 0x5c, 0xd1, 0xbc, 0xfd,
        0xee, 0x9b, 0xb9, 0xe5, 0xc4, 0xc0, 0xa8, 0x77,
        0x67, 0xa4, 0xd1, 0x95, 0x34, 0xe4, 0x6d, 0x5f,
        0x25, 0x02, 0x8d, 0x65, 0xdd, 0x11, 0x63, 0x55,
        0x04, 0x01, 0x21, 0x60, 0xc1, 0x5c, 0xef, 0x77,
        0x33, 0x01, 0x1c, 0xa2, 0x11, 0x2b, 0xdd, 0x2b,
        0x74, 0x99, 0x23, 0x38, 0x05, 0x1b, 0x7e, 0x2e,
        0x01, 0x52, 0xfe, 0x9c, 0x23, 0xde, 0x3e, 0x1a,
        0x72, 0xf4, 0xff, 0x7b, 0x02, 0xaa, 0x08, 0xcf,
        0xe0, 0x5b, 0x83, 0xbe, 0x85, 0x5a, 0xe8, 0x9d,
        0x11, 0x3e, 0xff, 0x2f, 0xc6, 0x97, 0x67, 0x36,
        0x6c, 0x0f, 0x81, 0x9c, 0x26, 0x29, 0xb1, 0x0f,
        0xbb, 0x53, 0xbd, 0xf4, 0xec, 0x2a, 0x84, 0x41,
        0x28, 0x3b, 0x86, 0x40, 0x95, 0x69, 0x55, 0x5f,
        0x30, 0xee, 0xda, 0x1e, 0x6c, 0x4b, 0x25, 0xd6,
        0x2f, 0x2c, 0x0e, 0x3c, 0x1a, 0x26, 0xa0, 0x3e,
        0xef, 0x09, 0xc6, 0x2b, 0xe5, 0xa1, 0x0c, 0x03,
        0xa8, 0xf5, 0x39, 0x70, 0x31, 0xc4, 0x32, 0x79,
        0xd1, 0xd9, 0xc2, 0xcc, 0x32, 0x4a, 0xf1, 0x2f,
        0x57, 0x5a, 0xcc, 0xe5, 0xc3, 0xc5, 0xd5, 0x4e,
        0x86, 0x56, 0xca, 0x64, 0xdb, 0xab, 0x61, 0x85,
        0x8f, 0xf9, 0x20, 0x02, 0x40, 0x66, 0x76, 0x9e,
        0x5e, 0xd4, 0xac, 0xf0, 0x47, 0xa6, 0x50, 0x5f,
        0xc2, 0xaf, 0x55, 0x9b, 0xa3, 0xc9, 0x8b, 0xf8,
        0x42, 0xd5, 0xcf, 0x1a, 0x95, 0x22, 0xd9, 0xd1,
        0x0b, 0x92, 0x51, 0xca, 0xde, 0x46, 0x02, 0x0d,
        0x8b, 0xee, 0xd9, 0xa0, 0x04, 0x74, 0xf5, 0x0e,
        0xb0, 0x3a, 0x62, 0xec, 0x3c, 0x91, 0x29, 0x33,
        0xa7, 0x78, 0x22, 0x92, 0xac, 0x27, 0xe6, 0x2d,
        0x6f, 0x56, 0x8a, 0x5d, 0x72, 0xc2, 0xf1, 0x5c,
        0x54, 0x11, 0x97, 0x24, 0x61, 0xcb, 0x0c, 0x52,
        0xd4, 0x57, 0x56, 0x22, 0x86, 0xf0, 0x19, 0x27,
        0x76, 0x30, 0x04, 0xf4, 0x39, 0x7b, 0x1a, 0x5a,
        0x04, 0x0d, 0xec, 0x59, 0x9a, 0x31, 0x4c, 0x40,
        0x19, 0x6d, 0x3c, 0x41, 0x1b, 0x0c, 0xca, 0xeb,
        0x25, 0x39, 0x6c, 0x96, 0xf8, 0x55, 0xd0, 0xec
    };

    /* 1024 byte shared secret */
    static const byte shared_secret[] = {
        0xf9, 0x55, 0x4f, 0x48, 0x38, 0x74, 0xb7, 0x46, 0xa3, 0xc4, 0x2e, 0x88, 0xf0, 0x34, 0xab, 0x1d,
        0xcd, 0xa5, 0x58, 0xa7, 0x95, 0x88, 0x36, 0x62, 0x6f, 0x8a, 0xbd, 0xf2, 0xfb, 0x6f, 0x3e, 0xb9,
        0x91, 0x65, 0x58, 0xef, 0x70, 0x2f, 0xd5, 0xc2, 0x97, 0x70, 0xcb, 0xce, 0x8b, 0x78, 0x1c, 0xe0,
        0xb9, 0xfa, 0x77, 0x34, 0xd2, 0x4a, 0x19, 0x58, 0x11, 0xfd, 0x93, 0x84, 0x40, 0xc0, 0x8c, 0x19,
        0x8b, 0x98, 0x50, 0x83, 0xba, 0xfb, 0xe2, 0xad, 0x8b, 0x81, 0x84, 0x63, 0x90, 0x41, 0x4b, 0xf8,
        0xe8, 0x78, 0x86, 0x04, 0x09, 0x8d, 0x84, 0xd1, 0x43, 0xfd, 0xa3, 0x58, 0x21, 0x2a, 0x3b, 0xb1,
        0xa2, 0x5b, 0x48, 0x74, 0x3c, 0xa9, 0x16, 0x34, 0x28, 0xf0, 0x8e, 0xde, 0xe2, 0xcf, 0x8e, 0x68,
        0x53, 0xab, 0x65, 0x06, 0xb7, 0x86, 0xb1, 0x08, 0x4f, 0x73, 0x97, 0x00, 0x10, 0x95, 0xd1, 0x84,
        0x72, 0xcf, 0x14, 0xdb, 0xff, 0xa7, 0x80, 0xd8, 0xe5, 0xf2, 0x2c, 0x89, 0x37, 0xb0, 0x81, 0x2c,
        0xf5, 0xd6, 0x7d, 0x1b, 0xb0, 0xe2, 0x8e, 0x87, 0x32, 0x3d, 0x37, 0x6a, 0x79, 0xaa, 0xe7, 0x08,
        0xc9, 0x67, 0x55, 0x5f, 0x1c, 0xae, 0xa6, 0xf5, 0xef, 0x79, 0x3a, 0xaf, 0x3f, 0x82, 0x14, 0xe2,
        0xf3, 0x69, 0x91, 0xed, 0xb7, 0x9e, 0xc9, 0xde, 0xd0, 0x29, 0x70, 0xd9, 0xeb, 0x0f, 0xf5, 0xc7,
        0xf6, 0x7c, 0xa7, 0x7f, 0xec, 0xed, 0xe1, 0xbd, 0x13, 0xe1, 0x43, 0xe4, 0x42, 0x30, 0xe3, 0x5f,
        0xe0, 0xf3, 0x15, 0x55, 0x2f, 0x7a, 0x42, 0x17, 0x67, 0xcb, 0xc2, 0x4f, 0xd0, 0x85, 0xfc, 0x6c,
        0xec, 0xe8, 0xfc, 0x25, 0x78, 0x4b, 0xe4, 0x0f, 0xd4, 0x3d, 0x78, 0x28, 0xd3, 0x53, 0x79, 0xcb,
        0x2c, 0x82, 0x67, 0x9a, 0xdc, 0x32, 0x55, 0xd2, 0xda, 0xae, 0xd8, 0x61, 0xce, 0xd6, 0x59, 0x0b,
        0xc5, 0x44, 0xeb, 0x08, 0x81, 0x8c, 0x65, 0xb2, 0xb7, 0xa6, 0xff, 0xf7, 0xbf, 0x99, 0xc6, 0x8a,
        0xbe, 0xde, 0xc2, 0x17, 0x56, 0x05, 0x6e, 0xd2, 0xf1, 0x1e, 0xa2, 0x04, 0xeb, 0x02, 0x74, 0xaa,
        0x04, 0xfc, 0xf0, 0x6b, 0xd4, 0xfc, 0xf0, 0x7a, 0x5f, 0xfe, 0xe2, 0x74, 0x7f, 0xeb, 0x9b, 0x6a,
        0x8a, 0x09, 0x96, 0x5d, 0xe1, 0x91, 0xb6, 0x9e, 0x37, 0xd7, 0x63, 0xd7, 0xb3, 0x5c, 0xb5, 0xa3,
        0x5f, 0x62, 0x00, 0xdf, 0xc5, 0xbf, 0x85, 0xba, 0xa7, 0xa9, 0xb6, 0x1f, 0x76, 0x78, 0x65, 0x01,
        0xfe, 0x1d, 0x6c, 0xfe, 0x15, 0x9e, 0xf4, 0xb1, 0xbc, 0x8d, 0xad, 0x3c, 0xec, 0x69, 0x27, 0x57,
        0xa4, 0x89, 0x77, 0x46, 0xe1, 0x49, 0xc7, 0x22, 0xde, 0x79, 0xe0, 0xf7, 0x3a, 0xa1, 0x59, 0x8b,
        0x59, 0x71, 0xcc, 0xd6, 0x18, 0x24, 0xc1, 0x8a, 0x2f, 0xe3, 0xdf, 0xdd, 0x6c, 0xf7, 0x62, 0xaa,
        0x15, 0xaa, 0x39, 0x37, 0x3b, 0xaf, 0x7d, 0x6e, 0x88, 0xeb, 0x19, 0xa8, 0xa0, 0x26, 0xd3, 0xaa,
        0x2d, 0xcc, 0x5f, 0x56, 0x99, 0x86, 0xa9, 0xed, 0x4d, 0x02, 0x31, 0x40, 0x97, 0x70, 0x83, 0xa7,
        0x08, 0x98, 0x7e, 0x49, 0x46, 0xd9, 0x75, 0xb5, 0x7a, 0x6a, 0x40, 0x69, 0xa0, 0x6d, 0xb2, 0x18,
        0xc0, 0xad, 0x88, 0x05, 0x02, 0x95, 0x6f, 0xf7, 0x8f, 0xcb, 0xa2, 0xe4, 0x7b, 0xab, 0x4a, 0x0f,
        0x9a, 0x1b, 0xef, 0xcc, 0xd1, 0x6a, 0x5d, 0x1e, 0x6a, 0x2a, 0x8b, 0x5b, 0x80, 0xbc, 0x5f, 0x38,
        0xdd, 0xaf, 0xad, 0x44, 0x15, 0xb4, 0xaf, 0x26, 0x1c, 0x1a, 0x4d, 0xa7, 0x4b, 0xec, 0x88, 0x33,
        0x24, 0x42, 0xb5, 0x0c, 0x9c, 0x56, 0xd4, 0xba, 0xa7, 0xb9, 0x65, 0xd5, 0x76, 0xb2, 0xbc, 0x16,
        0x8e, 0xfa, 0x0c, 0x7a, 0xc0, 0xa2, 0x2c, 0x5a, 0x39, 0x56, 0x7d, 0xe6, 0xf8, 0xa9, 0xf4, 0x49,
        0xd0, 0x50, 0xf2, 0x5e, 0x4b, 0x0a, 0x43, 0xe4, 0x9a, 0xbb, 0xea, 0x35, 0x28, 0x99, 0x84, 0x83,
        0xec, 0xc1, 0xa0, 0x68, 0x15, 0x9a, 0x2b, 0x01, 0x04, 0x48, 0x09, 0x11, 0x1b, 0xb6, 0xa4, 0xd8,
        0x03, 0xad, 0xb6, 0x4c, 0x9e, 0x1d, 0x90, 0xae, 0x88, 0x0f, 0x75, 0x95, 0x25, 0xa0, 0x27, 0x13,
        0xb7, 0x4f, 0xe2, 0x3e, 0xd5, 0x59, 0x1a, 0x7c, 0xde, 0x95, 0x14, 0x28, 0xd1, 0xde, 0x84, 0xe4,
        0x07, 0x7c, 0x5b, 0x06, 0xd6, 0xe6, 0x9c, 0x8a, 0xbe, 0xd2, 0xb4, 0x62, 0xd1, 0x67, 0x8a, 0x9c,
        0xac, 0x4f, 0xfa, 0x70, 0xd6, 0xc8, 0xc0, 0xeb, 0x5e, 0xf6, 0x3e, 0xdc, 0x48, 0x8e, 0xce, 0x3f,
        0x92, 0x3e, 0x60, 0x77, 0x63, 0x60, 0x6b, 0x76, 0x04, 0xa5, 0xba, 0xc9, 0xab, 0x92, 0x4e, 0x0d,
        0xdc, 0xca, 0x82, 0x44, 0x5f, 0x3a, 0x42, 0xeb, 0x01, 0xe7, 0xe0, 0x33, 0xb3, 0x32, 0xaf, 0x4b,
        0x81, 0x35, 0x2d, 0xb6, 0x57, 0x15, 0xfe, 0x52, 0xc7, 0x54, 0x2e, 0x41, 0x3b, 0x22, 0x6b, 0x12,
        0x72, 0xdb, 0x5c, 0x66, 0xd0, 0xb6, 0xb4, 0xfe, 0x90, 0xc0, 0x20, 0x34, 0x95, 0xf9, 0xe4, 0xc7,
        0x7e, 0x71, 0x89, 0x4f, 0x6f, 0xfb, 0x2a, 0xf3, 0xdf, 0x3f, 0xe3, 0xcf, 0x0e, 0x1a, 0xd9, 0xf2,
        0xc1, 0x02, 0x67, 0x5d, 0xdc, 0xf1, 0x7d, 0xe8, 0xcf, 0x64, 0x77, 0x4d, 0x12, 0x03, 0x77, 0x2c,
        0xfb, 0xe1, 0x59, 0xf7, 0x2c, 0x96, 0x9c, 0xaf, 0x46, 0x9c, 0xc7, 0x67, 0xcf, 0xee, 0x94, 0x50,
        0xc7, 0xa1, 0x23, 0xe6, 0x9f, 0x4d, 0x73, 0x92, 0xad, 0xf9, 0x4a, 0xce, 0xdb, 0x44, 0xd5, 0xe3,
        0x17, 0x05, 0x37, 0xdb, 0x9c, 0x6c, 0xc5, 0x7e, 0xb7, 0xd4, 0x11, 0x4a, 0x8c, 0x51, 0x03, 0xaa,
        0x73, 0x4b, 0x16, 0xd9, 0x79, 0xf5, 0xf1, 0x67, 0x20, 0x9b, 0x25, 0xe5, 0x41, 0x52, 0x59, 0x06,
        0x8b, 0xf2, 0x23, 0x2f, 0x6e, 0xea, 0xf3, 0x24, 0x0a, 0x94, 0xbb, 0xb8, 0x7e, 0xd9, 0x23, 0x4a,
        0x9f, 0x1f, 0xe1, 0x13, 0xb5, 0xfe, 0x85, 0x2f, 0x4c, 0xbe, 0x6a, 0x66, 0x02, 0x1d, 0x90, 0xd2,
        0x01, 0x25, 0x8a, 0xfd, 0x78, 0x3a, 0x28, 0xb8, 0x18, 0xc1, 0x38, 0x16, 0x21, 0x6b, 0xb4, 0xf9,
        0x64, 0x0f, 0xf1, 0x73, 0xc4, 0x5c, 0xd1, 0x41, 0xf2, 0xfe, 0xe7, 0x26, 0xad, 0x79, 0x12, 0x75,
        0x49, 0x48, 0xdb, 0x21, 0x71, 0x35, 0xf7, 0xb7, 0x46, 0x5a, 0xa1, 0x81, 0x25, 0x47, 0x31, 0xea,
        0x1d, 0x76, 0xbb, 0x32, 0x5a, 0x90, 0xb0, 0x42, 0x1a, 0x47, 0xe8, 0x0c, 0x82, 0x92, 0x43, 0x1c,
        0x0b, 0xdd, 0xe5, 0x25, 0xce, 0xd3, 0x06, 0xcc, 0x59, 0x5a, 0xc9, 0xa0, 0x01, 0xac, 0x29, 0x12,
        0x31, 0x2e, 0x3d, 0x1a, 0xed, 0x3b, 0xf3, 0xa7, 0xef, 0x52, 0xc2, 0x0d, 0x18, 0x1f, 0x03, 0x28,
        0xc9, 0x2b, 0x38, 0x61, 0xa4, 0x01, 0xc9, 0x3c, 0x11, 0x08, 0x14, 0xd4, 0xe5, 0x31, 0xe9, 0x3c,
        0x1d, 0xad, 0xf8, 0x76, 0xc4, 0x84, 0x9f, 0xea, 0x16, 0x61, 0x3d, 0x6d, 0xa3, 0x32, 0x31, 0xcd,
        0x1c, 0xca, 0xb8, 0x74, 0xc2, 0x45, 0xf3, 0x01, 0x9c, 0x7a, 0xaf, 0xfd, 0xe7, 0x1e, 0x5a, 0x18,
        0xb1, 0x9d, 0xbb, 0x7a, 0x2d, 0x34, 0x40, 0x17, 0x49, 0xad, 0x1f, 0xeb, 0x2d, 0xa2, 0x26, 0xb8,
        0x16, 0x28, 0x4b, 0x72, 0xdd, 0xd0, 0x8d, 0x85, 0x4c, 0xdd, 0xf8, 0x57, 0x48, 0xd5, 0x1d, 0xfb,
        0xbd, 0xec, 0x11, 0x5d, 0x1e, 0x9c, 0x26, 0x81, 0xbf, 0xf1, 0x16, 0x12, 0x32, 0xc3, 0xf3, 0x07,
        0x0e, 0x6e, 0x7f, 0x17, 0xec, 0xfb, 0xf4, 0x5d, 0xe2, 0xb1, 0xca, 0x97, 0xca, 0x46, 0x20, 0x2d,
        0x09, 0x85, 0x19, 0x25, 0x89, 0xa8, 0x9b, 0x51, 0x74, 0xae, 0xc9, 0x1b, 0x4c, 0xb6, 0x80, 0x62
    };

    rc = linuxkm_test_kpp_driver(WOLFKM_FFDHE8192_DRIVER,
                                 b_pub, expected_a_pub, sizeof(b_pub),
                                 secret, sizeof(secret),
                                 shared_secret, sizeof(shared_secret));

    return rc;
}
#endif /* HAVE_FFDHE_8192 */

/**
 * driver          the crypto driver to test
 * b_pub           pub from party B
 * expected_a_pub  the expected generated pub from party A
 * pub_len         pub key length
 * a_secret        the secret data for party A
 * secret_len      the secret data length
 * shared_secret   the expected generated shared secret
 * shared_s_len    the expected shared secret length
 * */
static int linuxkm_test_kpp_driver(const char * driver,
                                   const byte * b_pub,
                                   const byte * expected_a_pub,
                                   word32 pub_len,
                                   const byte * a_secret,
                                   word32 secret_len,
                                   const byte * shared_secret,
                                   word32 shared_s_len)
{
    int                  test_rc = WC_NO_ERR_TRACE(WC_FAILURE);
    struct crypto_kpp *  tfm = NULL;
    struct kpp_request * req = NULL;
    struct scatterlist   src, dst;
    int                  err = 0;
    byte *               src_buf = NULL;
    byte *               dst_buf = NULL;
    unsigned int         src_len = pub_len;
    unsigned int         dst_len = 0;
    /**
     * Allocate the kpp transform, and set up
     * the kpp request.
     * */
    tfm = crypto_alloc_kpp(driver, 0, 0);
    if (IS_ERR(tfm)) {
        pr_err("error: allocating kpp algorithm %s failed: %ld\n",
               driver, PTR_ERR(tfm));
        if (PTR_ERR(tfm) == -ENOMEM)
            test_rc = MEMORY_E;
        else
            test_rc = BAD_FUNC_ARG;
        tfm = NULL;
        goto test_kpp_end;
    }

    req = kpp_request_alloc(tfm, GFP_KERNEL);
    if (IS_ERR(req)) {
        pr_err("error: allocating kpp request %s failed\n",
               driver);
        if (PTR_ERR(req) == -ENOMEM)
            test_rc = MEMORY_E;
        else
            test_rc = BAD_FUNC_ARG;
        req = NULL;
        goto test_kpp_end;
    }

    err = crypto_kpp_set_secret(tfm, a_secret, secret_len);
    if (err) {
        pr_err("error: crypto_kpp_set_secret returned: %d\n", err);
        test_rc = BAD_FUNC_ARG;
        goto test_kpp_end;
    }

    /* large enough to hold largest req output. */
    dst_len = crypto_kpp_maxsize(tfm);
    if (dst_len <= 0) {
        pr_err("error: crypto_kpp_maxsize returned: %d\n", dst_len);
        test_rc = BAD_FUNC_ARG;
        goto test_kpp_end;
    }

    dst_buf = malloc(dst_len);
    if (dst_buf == NULL) {
        pr_err("error: allocating out buf failed");
        test_rc = MEMORY_E;
        goto test_kpp_end;
    }

    memset(dst_buf, 0, dst_len);

    /* generate pub key from input, and verify matches expected. */
    kpp_request_set_input(req, NULL, 0);
    sg_init_one(&dst, dst_buf, dst_len);
    kpp_request_set_output(req, &dst, dst_len);

    err = crypto_kpp_generate_public_key(req);
    if (err) {
        pr_err("error: crypto_kpp_generate_public_key returned: %d", err);
        test_rc = BAD_FUNC_ARG;
        goto test_kpp_end;
    }

    if (memcmp(expected_a_pub, sg_virt(req->dst), pub_len)) {
        pr_err("error: crypto_kpp_generate_public_key: wrong output");
        test_rc = WC_KEY_MISMATCH_E;
        goto test_kpp_end;
    }

    src_buf = malloc(src_len);
    if (src_buf == NULL) {
        pr_err("error: allocating in buf failed");
        test_rc = MEMORY_E;
        goto test_kpp_end;
    }

    memcpy(src_buf, b_pub, pub_len);

    /* generate shared secret, verify matches expected value. */
    sg_init_one(&src, src_buf, src_len);
    sg_init_one(&dst, dst_buf, dst_len);
    kpp_request_set_input(req, &src, src_len);
    kpp_request_set_output(req, &dst, dst_len);

    err = crypto_kpp_compute_shared_secret(req);
    if (err) {
        pr_err("error: crypto_kpp_compute_shared_secret returned: %d", err);
        test_rc = BAD_FUNC_ARG;
        goto test_kpp_end;
    }

    if (memcmp(shared_secret, sg_virt(req->dst), shared_s_len)) {
        pr_err("error: shared secret does not match");
        test_rc = BAD_FUNC_ARG;
        goto test_kpp_end;
    }

    test_rc = 0;
test_kpp_end:
    if (req) { kpp_request_free(req); req = NULL; }
    if (tfm) { crypto_free_kpp(tfm); tfm = NULL; }

    if (src_buf) { free(src_buf); src_buf = NULL; }
    if (dst_buf) { free(dst_buf); dst_buf = NULL; }

    #ifdef WOLFKM_DEBUG_DH
    pr_info("info: %s: self test returned: %d\n", driver, test_rc);
    #endif /* WOLFKM_DEBUG_DH */

    return test_rc;
}

#endif /* LINUXKM_LKCAPI_REGISTER_DH */

#endif /* !WC_SKIP_INCLUDED_C_FILES */
