/****************************************************************************
 *    lib/c/Formula.cpp - This file is part of coala						*
 *																			*
 *    Copyright (C) 2009  Torsten Grote										*
 *																			*
 *    This program is free software; you can redistribute it and/or modify	*
 *    it under the terms of the GNU General Public License as published by	*
 *    the Free Software Foundation; either version 3 of the License, or		*
 *    (at your option) any later version.									*
 *																			*
 *    This program is distributed in the hope that it will be useful,		*
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of		*
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *    GNU General Public License for more details.							*
 *																			*
 *    You should have received a copy of the GNU General Public License		*
 *    along with this program; if not, see http://www.gnu.org/licenses		*
 ****************************************************************************/

#include "Formula.h"
#include "FluentAction.h"

using namespace C;

Formula::Formula() {
	I_ = new vector<Formula*>();
	type_ = '?';
}

Formula::Formula(char type, Formula* identifier) {
	type_ = type;
	I_ = new vector<Formula*>(1, identifier);
}

Formula::Formula(char type, vector<Formula*>* vec) {
	type_ = type;
	if(vec == NULL) I_ = new vector<Formula*>();
	else I_ = vec;
}

Formula::~Formula () {
	delete I_;
}

void Formula::addPart(Formula* identifier) {
	I_->push_back(identifier);
}

char Formula::getType() {
	return type_;
}

Formula* Formula::getFirstFormula() {
	return I_->at(0);
}

vector<Formula*>::iterator Formula::begin() {
	return I_->begin();
}

vector<Formula*>::iterator Formula::end() {
	return I_->end();
}

void Formula::insert(vector<Formula*>::iterator loc, vector<Formula*>::iterator begin, vector<Formula*>::iterator end) {
	I_->insert(loc, begin, end);
}

Formula* Formula::invert() {
	if(getType() == 'c') {
		for(vector<Formula*>::iterator n = I_->begin(); n != I_->end(); ++n) {
	        *n = (*n)->invert();
	    }
		return this;
	}
	else if(getType() == '_') {
		return new Formula('n', new vector<Formula*>(1, this));
	}
	else if(getType() == 'n') {
		return (Formula*) I_->at(0);
	}
	else {
		return this;
	}
}

bool Formula::isTrue() {
	if(I_->size() == 1 && I_->at(0)->getType() == '_' && ((FluentAction*)I_->at(0))->getName() == "0") return true;
	else return false;
}

bool Formula::isFalse() {
	if(getType() == 'n' && I_->size() == 1 && I_->at(0)->getType() == '_' && ((FluentAction*)I_->at(0))->getName() == "0") return true;
	else return false;
}

FluentAction* Formula::getFluentAction() {
	if(getType() == '_') return (FluentAction*) this;
	else if(getType() == 'n' && I_->size() == 1 && I_->at(0)->getType() == '_') return (FluentAction*) I_->at(0);
	else throw std::runtime_error("getFluentAction() was called in wrong context. Please notify the author!");
}

// TODO check if this can be generalized for every Type of Formula (FluentAction as well)
void Formula::print(Printer* p, set<Variable*>* vars, string T) {
	assert(p->no_direct_enc);
	
	if(I_->empty()) {
		p->add("\"true\"");
	}
	else if(I_->size() == 1) {
		FluentAction* f;
		bool neg;

		if(I_->at(0)->getType() == 'n') {
			f = (FluentAction*) I_->at(0)->getFirstFormula();
			neg = true;
		}
		else {
			f = (FluentAction*) I_->at(0);
			neg = false;
		}
		
		if(neg)	p->add("neg(" + f->print(p, T) + ")");
		else	p->add(f->print(p, T));
		
		// get the vars
		set<Variable*>* tmp_vars = f->getVariables();
		if(!tmp_vars->empty())
			vars->insert(tmp_vars->begin(), tmp_vars->end());
		delete tmp_vars;
	}
	else {
		p->add("and(");
		for(vector<Formula*>::iterator n = I_->begin(); n != I_->end(); ++n) {
			FluentAction* f;
			bool neg;
			
			if((*n)->getType() == 'n') {
				f = (FluentAction*) (*n)->getFirstFormula();
				neg = true;
			}
			else {
				f = (FluentAction*) *n;
				neg = false;
			}
			
			if(neg)	p->add("neg(" + f->print(p, T) + ")");
			else	p->add(f->print(p, T));
			if(n != I_->end()-1) p->add(", ");
			
			// get the vars
			set<Variable*>* tmp_vars = f->getVariables();
			if(!tmp_vars->empty())
				vars->insert(tmp_vars->begin(), tmp_vars->end());
		}
		p->add(")");
	}
}

void Formula::printAlsoActions(Printer* p, set<Variable*>* vars, string T) {
	string F = "";
	string A = "";
	int f = 0;
	int a = 0;
	FluentAction* i;
	bool neg;
	
	for(vector<Formula*>::iterator n = I_->begin(); n != I_->end(); ++n) {
		// check for negation
		if((*n)->getType() == 'n') {
			i = (FluentAction*) (*n)->getFirstFormula();
			neg = true;
		}
		else {
			i = (FluentAction*) *n;
			neg = false;
		}
		
		if(i->getClass() == "Fluent") {
			if(f > 0) F += ", ";
			
			if(neg) {
				if(p->no_direct_enc) F += "neg(";
				else F += p->neg;
				F += i->print(p, T);
				if(p->no_direct_enc) F += ")";
			}
			else F += i->print(p, T);
			
			f++;
		}
		else {
			if(a > 0) A += ", ";
			
			if(neg) {
				if(p->no_direct_enc) A += "neg(";
				else A += p->neg;
				A += i->print(p, T);
				if(p->no_direct_enc) A += ")";
			}
			else A += i->print(p, T);
			
			a++;
		}
		// get the vars
		set<Variable*>* tmp_vars = i->getVariables();
		if(!tmp_vars->empty())
			vars->insert(tmp_vars->begin(), tmp_vars->end());
	}

	if(p->no_direct_enc) {
		if(f == 0)		F = "\"true\"";
		else if(f > 1)	F = "and("+F+")";
		
		if(a == 0)		A = "\"true\"";
		else if(a > 1)	A = "and("+A+")";
		
		p->add(F + ", " + A);
	}
	else {
		p->add(F);
		if(f != 0 && a != 0) p->add(", ");
		p->add(A);
	}
}

void Formula::printCompositeFluents(Printer* p, Types* types) {
	if(I_->size() > 1) {
		set<Variable*> vars;
		
		// print cfluent
		p->add("cfluent(");
		print(p, &vars);
		p->add(")");
		
		if(vars.empty()) p->add(".\n");
		else {
			p->add(" :- ");
			types->print(p, &vars, 0);
			p->add(".\n");
		}
	
		// print fpart
		for(vector<Formula*>::iterator n = I_->begin(); n != I_->end(); ++n) {
			FluentAction* f;
			bool neg;
			
			if((*n)->getType() == 'n') {
				f = (FluentAction*) (*n)->getFirstFormula();
				neg = true;
			}
			else {
				f = (FluentAction*) *n;
				neg = false;
			}
			
			p->add("fpart(");
			print(p, &vars);
			p->add(", ");
			if(neg)	p->add("neg(" + f->print(p, "") + ")");
			else	p->add(f->print(p, ""));
			p->add(")");

			if(vars.empty()) p->add(".\n");
			else {
				p->add(" :- ");
				types->print(p, &vars, 0);
				p->add(".\n");
			}
		}
	} // end if size>1
}

void Formula::printCompositeFluentActions(Printer* p, Types* types) {
	string F = "";
	vector<string> Fs = vector<string>();
	set<Variable*> F_vars;
	string A = "";
	vector<string> As = vector<string>();
	set<Variable*> A_vars;
	int f = 0;
	int a = 0;
	
	for(vector<Formula*>::iterator n = I_->begin(); n != I_->end(); ++n) {
		FluentAction* i;
		bool neg;
		
		if((*n)->getType() == 'n') {
			i = (FluentAction*) (*n)->getFirstFormula();
			neg = true;
		}
		else {
			i = (FluentAction*) *n;
			neg = false;
		}

		if(i->getClass() == "Fluent") {
			if(f > 0) F += ", ";
			f++;
			
			if(neg) {
				F += "neg(" + i->print(p, "") + ")";
				Fs.push_back("neg(" + i->print(p, "") + ")");
			}
			else {
				F += i->print(p, "");
				Fs.push_back(i->print(p, ""));
			}
			
			set<Variable*>* tmp_vars = i->getVariables();
			if(!tmp_vars->empty())
				F_vars.insert(tmp_vars->begin(), tmp_vars->end());
		}
		else {
			if(a > 0) A += ", ";
			a++;
			
			if(neg) {
				A += "neg(" + i->print(p, "") + ")";
				As.push_back("neg(" + i->print(p, "") + ")");
			}
			else {
				A += i->print(p, "");
				As.push_back(i->print(p, ""));
			}

			set<Variable*>* tmp_vars = i->getVariables();
			if(!tmp_vars->empty())
				A_vars.insert(tmp_vars->begin(), tmp_vars->end());
		}
	}
	
	if(f > 1) {
		p->add("cfluent(and("+F+"))");
		
		if(F_vars.empty()) p->add(".\n");
		else {
			p->add(" :- ");
			types->print(p, &F_vars, 0);
			p->add(".\n");
		}
		
		for(vector<string>::iterator n = Fs.begin(); n != Fs.end(); ++n) {
			p->add("fpart(and("+F+"), "+*n+")");
			
			if(F_vars.empty()) p->add(".\n");
			else {
				p->add(" :- ");
				types->print(p, &F_vars, 0);
				p->add(".\n");
			}
		}
	}
	
	if(a > 1) {
		p->add("caction(and("+A+"))");
		
		if(A_vars.empty()) p->add(".\n");
		else {
			p->add(" :- ");
			types->print(p, &A_vars, 0);
			p->add(".\n");
		}
		
		for(vector<string>::iterator n = As.begin(); n != As.end(); ++n) {
			p->add("apart(and("+A+"), "+*n+")");
			
			if(A_vars.empty()) p->add(".\n");
			else {
				p->add(" :- ");
				types->print(p, &A_vars, 0);
				p->add(".\n");
			}
		}
	}
}

