/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL ES-CM 1.1 plugin
 *
 * Copyright © 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 *         Sergi Alvarez <salvarez@fluendo.com>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#if HAVE_GDL
#include <glib/gprintf.h>
#include "libgdl.h"
#include "EGL/egl.h"
#define DEFAULT_WIDTH 1280
#define DEFAULT_HEIGHT 720
#endif

#include <string.h>
#include <unistd.h>
#include <fcntl.h>

#include "pgmgleseglbackend.h"
#include "pgmglesviewport.h"

/*  */
typedef struct {
  EGLint       error;
  const gchar *string;
} EglError;

/*  */
static EglError egl_error_map[] = {
  { EGL_NOT_INITIALIZED,     "not initialized"     },
  { EGL_BAD_ACCESS,          "bad access"          },
  { EGL_BAD_ALLOC,           "bad allocation"      },
  { EGL_BAD_ATTRIBUTE,       "bad attribute"       },
  { EGL_BAD_CONFIG,          "bad configuration"   },
  { EGL_BAD_CONTEXT,         "bad context"         },
  { EGL_BAD_CURRENT_SURFACE, "bad current surface" },
  { EGL_BAD_DISPLAY,         "bad display"         },
  { EGL_BAD_MATCH,           "bad match"           },
  { EGL_BAD_NATIVE_PIXMAP,   "bad native pixmap"   },
  { EGL_BAD_NATIVE_WINDOW,   "bad native window"   },
  { EGL_BAD_PARAMETER,       "bad parameter"       },
  { EGL_BAD_SURFACE,         "bad surface"         },
  { 0,                       "unknown error"       }
};

#ifdef HAVE_TSLIB

/* Prototypes */
static gboolean event_prepare  (GSource *source,
                                gint *timeout);
static gboolean event_check    (GSource *source);
static gboolean event_dispatch (GSource *source,
                                GSourceFunc callback,
                                gpointer data);

/* Touch screen event handling functions */
static GSourceFuncs event_funcs = {
  event_prepare, event_check, event_dispatch, NULL
};

#endif /* HAVE_TSLIB */

GST_DEBUG_CATEGORY_STATIC (pgm_gles_eglbackend_debug);
#define GST_CAT_DEFAULT pgm_gles_eglbackend_debug

static PgmGlesBackendClass *parent_class = NULL;

/* Private functions */

/* */
static gboolean
get_egl_error (const gchar *function)
{
  guint i = 0;
  EGLint error;

  error = eglGetError ();

  if (error == EGL_SUCCESS)
    return TRUE;

  /* Search for the error */
  while (egl_error_map[i].error && error != egl_error_map[i].error)
    i++;

  /* Then log it */
  GST_ERROR ("%s failed: %s\n", function, egl_error_map[i].string);

  return FALSE;
}

#ifdef HAVE_TSLIB

/* Events preparing */
static gboolean
event_prepare (GSource *source,
               gint *timeout)
{
  PgmGlesEglBackendSource *eglbackend_source =
    (PgmGlesEglBackendSource *) source;
  PgmGlesBackend *backend = PGM_GLES_BACKEND (eglbackend_source->eglbackend);
  gint requested_fps = backend->context->requested_fps;

  /* If there are rendering updates */
  if (backend->context->update_tag)
    {
      /* If the rendering is blocked on the vblank, we give a timeout of 0 */
      if (requested_fps == 0)
        *timeout = 0;
      /* If the rendering is done in a timeout, we use the same timeout */
      else
        *timeout = requested_fps;
    }
  /* If there are no rendering updates, we poll with no timeout */
  else
    *timeout = -1;

  return FALSE;
}

/* Events checking */
static gboolean
event_check (GSource *source)
{
  PgmGlesEglBackendSource *eglbackend_source =
    (PgmGlesEglBackendSource *) source;

  return (eglbackend_source->poll_fd.revents & G_IO_IN) != 0;
}

/* Events dispatching */
static gboolean
event_dispatch (GSource *source,
                GSourceFunc callback,
                gpointer data)
{
  PgmGlesEglBackendSource *eglbackend_source =
    (PgmGlesEglBackendSource*) source;
  PgmGlesBackend *backend = PGM_GLES_BACKEND (eglbackend_source->eglbackend);
  PgmGlesViewport *glesviewport = backend->context->glesviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glesviewport);
  static gboolean clicked = FALSE;
  static gint last_x = -1;
  static gint last_y = -1;
  struct ts_sample sample;

  /* Get the event */
  while (ts_read (eglbackend_source->ts_device, &sample, 1) > 0)
    {
      GST_DEBUG_OBJECT (eglbackend_source->eglbackend, "touchscreen event: "
                        "(time=%ld.%06ld, x=%d y=%d pressure=%d)",
                        sample.tv.tv_sec, sample.tv.tv_usec, sample.x,
                        sample.y, sample.pressure);

      /* Build the event */
      if (sample.pressure)
        {
          /* Press */
          if (!clicked)
            {
              PgmEventButton *pgmevent;
              pgmevent = (PgmEventButton*) pgm_event_new (PGM_BUTTON_PRESS);
              pgmevent->time = sample.tv.tv_sec * 1000
                + sample.tv.tv_usec * 0.001f;
              pgmevent->pressure = sample.pressure;
              pgmevent->button = PGM_BUTTON_LEFT;
              pgmevent->x = last_x = sample.x;
              pgmevent->y = last_y = sample.y;
              clicked = TRUE;
              pgm_viewport_push_event (viewport, (PgmEvent*) pgmevent);
            }
          /* Pressure */
          else if (last_x == sample.x && last_y == sample.y)
            {
              PgmEventButton *pgmevent;
              pgmevent = (PgmEventButton*) pgm_event_new (PGM_BUTTON_PRESSURE);
              pgmevent->time = sample.tv.tv_sec * 1000
                + sample.tv.tv_usec * 0.001f;
              pgmevent->pressure = sample.pressure;
              pgmevent->button = PGM_BUTTON_LEFT;
              pgmevent->x = last_x = sample.x;
              pgmevent->y = last_y = sample.y;
              pgm_viewport_push_event (viewport, (PgmEvent*) pgmevent);
            }
          /* Motion */
          else
            {
              PgmEventMotion *pgmevent;
              pgmevent = (PgmEventMotion*) pgm_event_new (PGM_MOTION_NOTIFY);
              pgmevent->time = sample.tv.tv_sec * 1000
                + sample.tv.tv_usec * 0.001f;
              pgmevent->pressure = sample.pressure;
              pgmevent->x = last_x = sample.x;
              pgmevent->y = last_y = sample.y;
              pgm_viewport_push_event (viewport, (PgmEvent*) pgmevent);
            }
        }
      else
        {
          PgmEventButton *pgmevent;
          pgmevent = (PgmEventButton*) pgm_event_new (PGM_BUTTON_RELEASE);
          pgmevent->time = sample.tv.tv_sec * 1000
            + sample.tv.tv_usec * 0.001f;
          pgmevent->pressure = sample.pressure;
          pgmevent->button = PGM_BUTTON_LEFT;
          pgmevent->x = sample.x;
          pgmevent->y = sample.y;
          clicked = FALSE;
          last_x = -1;
          last_y = -1;
          pgm_viewport_push_event (viewport, (PgmEvent*) pgmevent);
        }
    }

  return TRUE;
}

/* Add touch screen event handling source to the rendering main context */
static gboolean
add_event_source (PgmGlesEglBackend *eglbackend)
{
  PgmGlesBackend *glesbackend = PGM_GLES_BACKEND (eglbackend);
  GMainContext *render_context = glesbackend->context->render_context;
  PgmGlesEglBackendSource *source;
  const gchar *device_string;
  struct tsdev *ts_device;
  gint flags;
  gint fd;

  /* Get the device name in the environment, open and configure it */
  device_string = g_getenv ("TSLIB_TSDEVICE");
  ts_device = ts_open (device_string, 0);
  if (!ts_device)
    {
      GST_WARNING_OBJECT (eglbackend, "could not open touch screen device: %s",
                          device_string);
      return FALSE;
    }
  if (ts_config (ts_device))
    {
      GST_WARNING_OBJECT (eglbackend, "could not configure touch screen device:"
                          "%s", device_string);
      ts_close (ts_device);
      return FALSE;
    }
  GST_DEBUG_OBJECT (eglbackend, "configured touch screen device: %s",
                    device_string);

  /* The touch screen device file descriptor needs to be non-blocking so that
   * we can flush completely the received events. If we are not doing that, the
   * release events are not sent when the stylus releases the touchscreen but
   * at the next pressure. */
  fd = ts_fd (ts_device);
  flags = fcntl (fd, F_GETFL, 0);
  fcntl (fd, F_SETFL, flags | O_NONBLOCK);

  /* Create and initialize the touch screen event handling dedicated source */
  source = (PgmGlesEglBackendSource *)
    g_source_new (&event_funcs, sizeof (PgmGlesEglBackendSource));
  source->poll_fd.fd = fd;
  source->poll_fd.events = G_IO_IN;
  source->ts_device = ts_device;
  source->eglbackend = eglbackend;
  g_source_add_poll ((GSource *) source, &source->poll_fd);

  /* Attach it */
  g_source_set_can_recurse ((GSource *) source, TRUE);
  g_source_set_priority ((GSource *) source, G_PRIORITY_DEFAULT-20);
  eglbackend->event_tag = g_source_attach ((GSource *) source, render_context);

  return TRUE;
}

/* Remove touch screen event handling source from the rendering main context */
static void
remove_event_source (PgmGlesEglBackend *eglbackend)
{
  PgmGlesBackend *glesbackend = PGM_GLES_BACKEND (eglbackend);
  GMainContext *render_context = glesbackend->context->render_context;
  GSource *source;

  /* Get the source from the rendering GMainContext */
  source = g_main_context_find_source_by_id (render_context,
                                             eglbackend->event_tag);

  /* And remove it */
  if (source)
    {
      ts_close (((PgmGlesEglBackendSource*) source)->ts_device);
      g_source_destroy (source);
      g_source_unref ((GSource *) source);
    }
}

#endif /* HAVE_TSLIB */

/* PgmGlesBackend methods */
#if HAVE_GDL
#include <signal.h>

/* Global pointer to the last initialization of the GLES EGL backend */
static PgmGlesEglBackend *gbackend = NULL;

static void
sigint_handler(int sig)
{
  static int sigint_called = 0;
  g_fprintf (stderr, "\n^C\n");
  if (sigint_called++)
    return;
  if (gbackend != NULL) {
    eglDestroyContext (gbackend->display, gbackend->context);
    gbackend->context = NULL;
    eglDestroySurface (gbackend->display, gbackend->surface);
    gbackend->surface = NULL;
    eglMakeCurrent (gbackend->display, EGL_NO_SURFACE, EGL_NO_SURFACE,
                      EGL_NO_CONTEXT);
    eglTerminate (gbackend->display);
    gbackend->display = NULL;
    gdl_close();
  }
  g_fprintf(stderr, "GDL terminated\n");
  exit(1);
}

/* GDL get/set tvmode resolution */

static inline gboolean
gdl_tvmode_get_size (gint * width, gint * height)
{
  gboolean ret = FALSE;
  gdl_display_info_t di;
  memset (&di, 0, sizeof (di));

  /* Read current hardware state */
  if (gdl_get_display_info (GDL_DISPLAY_ID_0, &di) != GDL_SUCCESS) {
    GST_ERROR ("gdl_get_display_info failed");
    goto beach;
  }

  *width = di.tvmode.width;
  *height = di.tvmode.height;

  ret = TRUE;
beach:
  return ret;
}

static int
initialize_gdl(PgmGlesEglBackend *glesbackend, gdl_plane_id_t plane)
{
  PgmGlesBackend *backend = PGM_GLES_BACKEND (glesbackend);
  PgmGlesViewport *glesviewport = backend->context->glesviewport;
  gdl_display_info_t di;
  PgmViewport *viewport = PGM_VIEWPORT (glesviewport);
  gdl_pixel_format_t pixelFormat = GDL_PF_ARGB_32;
  gdl_color_space_t colorSpace = GDL_COLOR_SPACE_RGB;
  gdl_rectangle_t rect;
  gboolean b_true = TRUE;
  gdl_ret_t rc = GDL_SUCCESS;
  gint width, height;
  int ret = TRUE;

  gdl_init(0);
  signal(SIGINT, sigint_handler);

  if (!gdl_tvmode_get_size (&width, &height)) {
    /* Write current hardware state */
    di.id = GDL_DISPLAY_ID_0;
    di.gamma = GDL_GAMMA_LINEAR;
    di.tvmode.refresh = GDL_REFRESH_50;
    di.tvmode.interlaced = 0;
    di.tvmode.width = DEFAULT_WIDTH;
    di.tvmode.height = DEFAULT_HEIGHT;
    di.color_space = (di.tvmode.width>720) ? 
      GDL_COLOR_SPACE_BT709 : GDL_COLOR_SPACE_BT601;
    if (gdl_set_display_info (&di) != GDL_SUCCESS) {
      GST_ERROR ("gdl_set_display_info failed");
      goto beach;
    }
    gdl_port_set_attr (GDL_PD_ID_HDMI, GDL_PD_ATTR_ID_POWER, &b_true);
    gdl_port_set_attr (GDL_PD_ID_HDMI, GDL_PD_ATTR_ID_PAR, &b_true);

    if (!gdl_tvmode_get_size (&width, &height))
      ret = FALSE;
    if (ret == FALSE) {
      goto beach;
    }
  }

  /* force viewport size to fullscreen */
  viewport->width = width;
  viewport->height = height;
  viewport->projected_width = width;
  viewport->projected_height = height;
  glesviewport->projected_w = width;
  glesviewport->projected_h = height;

  rc = gdl_plane_reset(plane);
  if (rc == GDL_SUCCESS)
    rc = gdl_plane_config_begin(plane);

  if (rc == GDL_SUCCESS)
    rc = gdl_plane_set_attr(GDL_PLANE_SRC_COLOR_SPACE, &colorSpace);

  if (rc == GDL_SUCCESS)
    rc = gdl_plane_set_attr(GDL_PLANE_PIXEL_FORMAT, &pixelFormat);

  rect.origin.x = 0;
  rect.origin.y = 0;
  rect.width = width;
  rect.height = height;
  if (rc == GDL_SUCCESS)
    rc = gdl_plane_set_attr(GDL_PLANE_DST_RECT, &rect);

  if (rc == GDL_SUCCESS)
    rc = gdl_plane_config_end(GDL_FALSE);
  else
    gdl_plane_config_end(GDL_TRUE);

  if (rc != GDL_SUCCESS)
    {
      GST_DEBUG_OBJECT (glesbackend,
        "GDL configuration failed! GDL error code is 0x%x\n", rc);
      ret = FALSE;
    }
beach:
  return ret;
}
#endif

static gboolean
pgm_gles_egl_backend_create_window (PgmGlesBackend *glesbackend)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (glesbackend);
  gint swap_interval = 1;
  const gchar *env_var;
  EGLConfig config = 0;
  gint num_configs = 0;
  gint egl_major, egl_minor;

  EGLint attrib[] =
    {
        EGL_BUFFER_SIZE,       EGL_DONT_CARE,
        EGL_DEPTH_SIZE,        0,
        EGL_RED_SIZE,          8,
        EGL_GREEN_SIZE,        8,
        EGL_BLUE_SIZE,         8,
	EGL_RENDERABLE_TYPE,   EGL_OPENGL_ES_BIT, 
        EGL_NONE
    };

  g_return_val_if_fail (eglbackend != NULL, FALSE);

  GST_DEBUG_OBJECT (eglbackend, "create_window");

#if HAVE_GDL
  if (initialize_gdl (eglbackend, GDL_PLANE_ID_UPP_C) == FALSE)
    {
      GST_ERROR_OBJECT (eglbackend, "Cannot initialize GDL backend");
      goto error_display;
    }
#endif

  /* Initialize EGL for the default display */
  eglbackend->display = eglGetDisplay ((NativeDisplayType) EGL_DEFAULT_DISPLAY);
  if (eglbackend->display == EGL_NO_DISPLAY)
    {
      GST_ERROR_OBJECT (eglbackend, "cannot get a display");
      goto error_display;
    }

  if (!eglInitialize (eglbackend->display, &egl_major, &egl_minor))
    {
      get_egl_error ("eglInitialize");
      goto error;
    }

  /* Get and log EGL informations */
  GST_DEBUG_OBJECT (eglbackend, "EGL init version: %d.%d", egl_major, egl_minor);

  eglbackend->vendor = eglQueryString (eglbackend->display, EGL_VENDOR);
  eglbackend->version = eglQueryString (eglbackend->display, EGL_VERSION);
  eglbackend->extensions = eglQueryString (eglbackend->display, EGL_EXTENSIONS);

  GST_DEBUG_OBJECT (eglbackend, "EGL vendor: %s", eglbackend->vendor);
  GST_DEBUG_OBJECT (eglbackend, "EGL version: %s", eglbackend->version);
  GST_DEBUG_OBJECT (eglbackend, "EGL extensions: %s", eglbackend->extensions);

  if (!eglChooseConfig (eglbackend->display, attrib, &config, 1, &num_configs))
    {
      get_egl_error ("eglChooseConfig");
      goto error;
    }
  GST_DEBUG_OBJECT (eglbackend, "EGL config %d (num=%d)", config, num_configs);

  /* Create the context and make it current */
#if HAVE_GDL
  eglbackend->surface = eglCreateWindowSurface (eglbackend->display, config,
      (NativeWindowType)GDL_PLANE_ID_UPP_C, NULL);
  gbackend = eglbackend;

  if (eglBindAPI(EGL_OPENGL_ES_API) == 0)
    {
      GST_DEBUG_OBJECT (eglbackend, "eglBindAPI: %s", eglbackend->extensions);
      goto error;
    }
  eglWaitClient();
#else
  eglbackend->surface = eglCreateWindowSurface (eglbackend->display, config,
                                                (NativeWindowType) 0, NULL);
#endif

  if (eglbackend->surface == EGL_NO_SURFACE \
    || !get_egl_error ("eglCreateWindowSurface"))
    goto error;

  eglbackend->context = eglCreateContext (eglbackend->display, config,
                                          NULL, NULL);

  if (eglbackend->context == EGL_NO_CONTEXT
      || !get_egl_error ("eglCreateContext"))
    {
      goto error;
    }

#ifdef HAVE_TSLIB
  /* Touch screen event source */
  if (!add_event_source (eglbackend))
    {
      GST_ERROR_OBJECT (eglbackend, "couldn't add event handling source\n");
      eglMakeCurrent (eglbackend->display, EGL_NO_SURFACE, EGL_NO_SURFACE,
                      EGL_NO_CONTEXT);
      goto error;
    }
#else
  eglMakeCurrent (eglbackend->display, eglbackend->surface,
                  eglbackend->surface, eglbackend->context);
  if (!get_egl_error ("eglMakeCurrent"))
    {
      goto error;
    }
#endif /* HAVE_TSLIB */

#if HAVE_GDL
  /* Get and store the window size, the resolution in this case is the same */
  gdl_tvmode_get_size (&eglbackend->resolution_width,
      &eglbackend->resolution_height);
  eglbackend->size_mm_width = eglbackend->resolution_width;
  eglbackend->size_mm_height = eglbackend->resolution_height;
#else
  /* Get and store the window size, the resolution in this case is the same */
  eglQuerySurface (eglbackend->display, eglbackend->surface, EGL_WIDTH,
                   &eglbackend->size_mm_width);
  eglbackend->resolution_width = eglbackend->size_mm_width;
  eglQuerySurface (eglbackend->display, eglbackend->surface, EGL_HEIGHT,
                   &eglbackend->size_mm_height);
  eglbackend->resolution_height = eglbackend->size_mm_height;
#endif

  /* Specify the minimum number of video frame periods per buffer swap, this
   * is actually how vertical synchronization is handled */
  env_var = g_getenv ("PGM_GLES_SWAP_INTERVAL");
  if (env_var)
    swap_interval = atoi (env_var);
  eglSwapInterval (eglbackend->display, swap_interval);
  GST_DEBUG_OBJECT (eglbackend, "using a swap interval of %d", swap_interval);

  eglbackend->created = TRUE;

  return TRUE;

error:
  if (eglbackend->context != EGL_NO_CONTEXT)
    {
      eglDestroyContext (eglbackend->display, eglbackend->context);
    }
  if (eglbackend->surface != EGL_NO_SURFACE)
    {
      eglDestroySurface (eglbackend->display, eglbackend->surface);
    }

  eglTerminate (eglbackend->display);

error_display:
#if HAVE_GDL
  gdl_close();
#endif
  return FALSE;
}

static gboolean
pgm_gles_egl_backend_destroy_window (PgmGlesBackend *glesbackend)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (glesbackend);

  GST_DEBUG_OBJECT (eglbackend, "destroy_window");

  if (eglbackend->created)
    {
#ifdef HAVE_TSLIB
      remove_event_source (eglbackend);
#endif /* HAVE_TSLIB */
#if 0
  /* this makes the hardware crash */
  if (eglbackend->context != EGL_NO_CONTEXT)
    {
      eglDestroyContext (eglbackend->display, eglbackend->context);
    }
  if (eglbackend->surface != EGL_NO_SURFACE)
    {
      eglDestroySurface (eglbackend->display, eglbackend->surface);
    }
      eglMakeCurrent (eglbackend->display, EGL_NO_SURFACE, EGL_NO_SURFACE,
                      EGL_NO_CONTEXT);
#endif
      eglTerminate (eglbackend->display);
      eglbackend->created = FALSE;
    }
#if HAVE_GDL
  gdl_close();
#endif
  return TRUE;
}

static gboolean
pgm_gles_egl_backend_set_visibility (PgmGlesBackend *glesbackend,
                                     gboolean visible)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (glesbackend);

  GST_DEBUG_OBJECT (eglbackend, "set_visibility");

  /* FIXME */

  return TRUE;
}

static void
pgm_gles_egl_backend_swap_buffers (PgmGlesBackend *glesbackend)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (glesbackend);

  eglSwapBuffers (eglbackend->display, eglbackend->surface);
}

static gpointer
pgm_gles_egl_backend_get_proc_address (PgmGlesBackend *glesbackend,
                                       const gchar *proc_name)
{
  return eglGetProcAddress (proc_name);
}

static void
pgm_gles_egl_backend_get_screen_size_mm (PgmGlesBackend *glesbackend,
                                         gint *width,
                                         gint *height)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (glesbackend);

  *width = eglbackend->size_mm_width;
  *height = eglbackend->size_mm_height;
}

static void
pgm_gles_egl_backend_set_screen_resolution (PgmGlesBackend *glesbackend,
                               gint width, gint height)
{
#if HAVE_GDL
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (glesbackend);
  gdl_rectangle_t rect;                                                    
  gint num_configs = 0;
  EGLConfig config = 0;
  EGLint attrib[] =
    {
        EGL_BUFFER_SIZE,       EGL_DONT_CARE,
        EGL_DEPTH_SIZE,        0,
        EGL_RED_SIZE,          8,
        EGL_GREEN_SIZE,        8,
        EGL_BLUE_SIZE,         8,
	EGL_RENDERABLE_TYPE,   EGL_OPENGL_ES_BIT, 
        EGL_NONE
    };

  eglbackend->size_mm_width =
  eglbackend->resolution_width = width;
  eglbackend->size_mm_height =
  eglbackend->resolution_height = height;

  eglMakeCurrent (eglbackend->display, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT);
  eglDestroySurface (eglbackend->display, eglbackend->surface);
  eglbackend->surface = NULL;

  rect.origin.x = 0;
  rect.origin.y = 0;
  rect.width = width;
  rect.height = height;
  gdl_plane_config_begin (GDL_PLANE_ID_UPP_C);
  gdl_plane_set_attr (GDL_PLANE_DST_RECT, &rect);
  gdl_plane_config_end (GDL_FALSE);

  if (!eglChooseConfig (eglbackend->display, attrib, &config, 1, &num_configs))
    {
      GST_WARNING_OBJECT (eglbackend, "Cannot get EGL configuration");
    }
  eglbackend->surface = eglCreateWindowSurface (eglbackend->display, config,
    (NativeWindowType)GDL_PLANE_ID_UPP_C, NULL);

  if (eglBindAPI (EGL_OPENGL_ES_API) == 0)
    {
      GST_WARNING_OBJECT (eglbackend, "Cannot bind OpenGL-ES from EGL");
    }
  eglMakeCurrent (eglbackend->display, eglbackend->surface,
                  eglbackend->surface, eglbackend->context);
#endif
}

static void
pgm_gles_egl_backend_get_screen_resolution (PgmGlesBackend *glesbackend,
                                            gint *width, gint *height)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (glesbackend);

  *width = eglbackend->resolution_width;
  *height = eglbackend->resolution_height;
}

/* GObject stuff */

PGM_DEFINE_DYNAMIC_TYPE (PgmGlesEglBackend, pgm_gles_egl_backend,
                         PGM_TYPE_GLES_BACKEND)

void
pgm_gles_egl_backend_register (GTypeModule *module)
{
  pgm_gles_egl_backend_register_type (module);
}

static void
pgm_gles_egl_backend_dispose (GObject *object)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (object);

  GST_DEBUG_OBJECT (eglbackend, "dispose");

  if (eglbackend->created)
    pgm_gles_egl_backend_destroy_window (PGM_GLES_BACKEND (eglbackend));

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_gles_egl_backend_class_init (PgmGlesEglBackendClass *klass)
{
  GObjectClass *gobject_class;
  PgmGlesBackendClass *glesbackend_class;

  GST_DEBUG_CATEGORY_INIT (pgm_gles_eglbackend_debug, "pgm_gles_eglbackend", 0,
                           "OpenGL ES plugin: PgmGlesEglBackend");

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  glesbackend_class = PGM_GLES_BACKEND_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_dispose);

  /* PgmGlesBackend virtual table */
  glesbackend_class->create_window =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_create_window);
  glesbackend_class->destroy_window =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_destroy_window);
  glesbackend_class->swap_buffers =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_swap_buffers);
  glesbackend_class->get_proc_address =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_get_proc_address);
  glesbackend_class->set_visibility =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_set_visibility);
  glesbackend_class->get_screen_size_mm =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_get_screen_size_mm);
  glesbackend_class->get_screen_resolution =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_get_screen_resolution);
  glesbackend_class->set_screen_resolution =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_set_screen_resolution);
}

static void
pgm_gles_egl_backend_class_finalize (PgmGlesEglBackendClass *klass)
{
  return;
}

static void
pgm_gles_egl_backend_init (PgmGlesEglBackend *eglbackend)
{
  GST_DEBUG_OBJECT (eglbackend, "init");

  eglbackend->display = 0;
  eglbackend->surface = 0;
  eglbackend->context = 0;
  eglbackend->created = FALSE;
}

/* Public methods */

PgmGlesBackend*
pgm_gles_egl_backend_new (PgmGlesContext *glescontext)
{
  PgmGlesBackend *glesbackend;

  glesbackend = g_object_new (PGM_TYPE_GLES_EGL_BACKEND, NULL);
  GST_DEBUG_OBJECT (PGM_GLES_EGL_BACKEND (glesbackend),
                    "created new eglbackend");

  glesbackend->context = glescontext;

  return glesbackend;
}
