// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


/*
  infodialogs.cc
  This file contains code for the dialog boxes in the StarPlot Stars menu,
  and for the informational popup window.
*/

#include <gtk/gtk.h>
#include "starplot.h"
#include <cstring>
using std::string;

// ---------------------------------------------------------------------------
// This function displays an error message popup.
void my_gtk_error (GtkWindow *parent_win, const char * error_msg)
{
  GtkWidget * dialog = gtk_message_dialog_new (parent_win,
      GTK_DIALOG_DESTROY_WITH_PARENT,
      GTK_MESSAGE_ERROR,
      GTK_BUTTONS_CLOSE,
      error_msg);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}

// ---------------------------------------------------------------------------
/* This function displays a popup window of information about the star: */

void my_gtk_star_popup(Star &s)
{
  GtkWidget *popup, *OK, *topbox, *expander, *separator,
	    *namebox, *altname, *label[5];
  StringList starinfo = s.GetInfo(globals::chart_rules);
  string Name, RA, Dec, Distance, Class, Mag;

  Name = starinfo[1];
  starstrings::addgreek(Name);
  if (globals::chart_rules.StarLabels == NUMBER_LABEL)
    Name = starinfo[0] + ". " + Name;
  if (globals::chart_rules.CelestialCoords) {
    RA = string(_("R.A.")) + ": " + starinfo[2];
    Dec = string(_("Dec.")) + ": " + starinfo[3];
  }
  else {
    RA = string(_("Long.")) + ": " + starinfo[2];
    Dec = string(_("Lat.")) + ": " + starinfo[3];
  }
  Distance = string(_("Distance")) + ": " + starstrings::distance_to_str(
      s.GetStarDistance(), globals::chart_rules.ChartUnits[1]);
  Class = string(_("Spectral Class")) + ": " + starinfo[5];
  Mag = string(_("Magnitude")) + ": " + starstrings::ftoa(s.GetStarMagnitude());
  
  popup = gtk_dialog_new ();
  gtk_window_set_resizable (GTK_WINDOW (popup), false);
  gtk_window_set_title (GTK_WINDOW (popup), _("StarPlot: Star Info"));
  gtk_window_set_position (GTK_WINDOW (popup), GTK_WIN_POS_MOUSE);
  
  topbox = gtk_vbox_new (false, 0);
  gtk_container_set_border_width (GTK_CONTAINER (topbox), 10);
  gtk_widget_show (topbox);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (popup)->vbox), topbox,
		      true, true, 0);

#if HAVE_GTK_EXPANDER_NEW
  // name
  expander = gtk_expander_new ((string("<big><b>") + Name.c_str()
      + "</b></big>").c_str());
  gtk_expander_set_use_markup (GTK_EXPANDER (expander), true);
  gtk_expander_set_expanded (GTK_EXPANDER (expander), false);

  // alternate names
  namebox = gtk_vbox_new (true, 0);
  gtk_widget_show (namebox);
  gtk_container_add (GTK_CONTAINER (expander), namebox);
  
  for (unsigned int i = 10; i < starinfo.size(); i++) {
    starstrings::addgreek(starinfo[i]);
    altname = gtk_label_new (starinfo[i].c_str());
    gtk_label_set_markup (GTK_LABEL (altname),
	(string("<b>") + starinfo[i] + string("</b>")).c_str());
    gtk_misc_set_alignment(GTK_MISC (altname), 0.0F, 0.0F);
    gtk_widget_show (altname);
    gtk_box_pack_start (GTK_BOX (namebox), altname, true, true, 0);
  }
#else // no GtkExpander widget, just show best-known name
  expander = gtk_label_new (Name.c_str());
#endif
  gtk_widget_show (expander);
  gtk_box_pack_start (GTK_BOX (topbox), expander, false, false, 0);
  
  separator = gtk_hseparator_new ();
  gtk_widget_show (separator);
  gtk_box_pack_start (GTK_BOX (topbox), separator, false, false, 5);
  
  // other info
  label[0] = gtk_label_new (RA.c_str());
  label[1] = gtk_label_new (Dec.c_str());
  label[2] = gtk_label_new (Distance.c_str());
  label[3] = gtk_label_new (Class.c_str());
  label[4] = gtk_label_new (Mag.c_str());
  
  for (unsigned int i = 0; i < 5; i++) {
    gtk_box_pack_start (GTK_BOX (topbox), label[i], false, false, 0);
    gtk_misc_set_alignment(GTK_MISC (label[i]), 0.0F, 0.0F);
    gtk_widget_show (label[i]);
  }

  my_gtk_popup_button (&OK, GTK_DIALOG (popup)->action_area, popup);
  gtk_window_set_focus (GTK_WINDOW (popup), OK);
  gtk_widget_show (popup);
}

// ---------------------------------------------------------------------------
/* Begin auxiliary types and functions for my_gtk_list_setup () */

enum elisttype { CHARTLIST, NUMBERED_CHARTLIST, SEARCHLIST };
enum ecolumntype { COL_INDEX, COL_NAME, COL_ALTNAME, COL_RA, COL_DEC,
		   COL_DISTANCE, COL_SPCLASS, COL_MAGNITUDE, NUM_COLS };

// Print doubles with at most 4 significant figures.  Modified from here:
// http://mail.gnome.org/archives/gtk-list/2003-April/msg00326.html
void
print_double_to_text(GtkTreeViewColumn *column,
		     GtkCellRenderer *cell,
		     GtkTreeModel *treemodel,
		     GtkTreeIter *iter,
		     void * data)
{
  GtkCellRendererText *cell_text = (GtkCellRendererText *)cell;
  double d;

  /* Free the previous (default) text of the column's renderer. */
  g_free (cell_text->text);
  /* Get the double value from the model. */
  gtk_tree_model_get (treemodel, iter, (long)data, &d, -1);
  /* Now we can format the value ourselves. */
  cell_text->text = g_strdup (starstrings::ftoa(d, 4).c_str());
}

// Sort function for latitude coordinate
// (in order to get strings starting with '-' before those starting
// with '+')
int
signed_coord_sort_func(GtkTreeModel * model,
		       GtkTreeIter * item1, GtkTreeIter * item2,
		       void * data)
{
  char * s1, * s2;
  int result;
  gtk_tree_model_get (model, item1, COL_DEC, &s1, -1);
  gtk_tree_model_get (model, item2, COL_DEC, &s2, -1);

  // N.B. we expect strings that look like "[+ -]NNh NNm NN.NNs"
  char c1 = s1[0], c2 = s2[0];
  result = std::strcmp(s1, s2);

  if (c1 == '-' && c2 == '-')
    result *= -1;
  else if (c1 == '-')
    result = -1;
  else if (c2 == '-')
    result = +1;
 
  std::free (s1);
  std::free (s2);
  return result;
}

// Sort function for spectral class
int
spec_class_sort_func(GtkTreeModel * model,
		     GtkTreeIter * item1, GtkTreeIter * item2,
		     void * data)
{
  char * s1, * s2;
  int M1, M2;
  double m1, m2, MK1, MK2;
  int result;
  
  gtk_tree_model_get (model, item1, COL_SPCLASS, &s1, -1);
  gtk_tree_model_get (model, item2, COL_SPCLASS, &s2, -1);

  SpecClass *c1 = new SpecClass(string(s1));
  SpecClass *c2 = new SpecClass(string(s2));
  std::free (s1);
  std::free (s2);
  
  c1->initialize();
  c2->initialize();
  M1 = SpecClass::class_to_int(c1->classmajor());
  M2 = SpecClass::class_to_int(c2->classmajor());
  if (M1 < 0) M1 = 999;
  if (M2 < 0) M2 = 999;
  result = M1 - M2;
  if (result) goto done;
  
  m1 = c1->classminor();
  m2 = c2->classminor();
  result = static_cast<int>(100 * (m1 - m2));
  if (result) goto done;
  
  MK1 = c1->MKtype();
  MK2 = c2->MKtype();
  result = static_cast<int>(100 * (MK1 - MK2));

 done:
  delete c1;
  delete c2;
  return result;
}

/* Here is a function which sets up the treeview for starinfo() and
 *  starsearch().  "box" is the box into which the treeview should
 *  be packed. */

void my_gtk_list_setup (GtkWidget *box, GtkTreeView **treeviewptr,
			enum elisttype listtype)
{
  GtkWidget *scrolled_window;
  GtkTreeView *treeview;
  GtkListStore *store;

  // set up a scroll window to hold the chart
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_window),
				       GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX (box), scrolled_window, true, true, 0);
  gtk_widget_show (scrolled_window);

  // set up the data model
  store = gtk_list_store_new (NUM_COLS,	      // number of possible columns
			      G_TYPE_UINT,    // star index number
			      G_TYPE_STRING,  // star name
			      G_TYPE_STRING,  // alternate name
			      G_TYPE_STRING,  // right ascension
			      G_TYPE_STRING,  // declination
			      G_TYPE_DOUBLE,  // distance
			      G_TYPE_STRING,  // spectral class
			      G_TYPE_DOUBLE); // absolute magnitude
  
  // set up the data viewer
  *treeviewptr = (GtkTreeView *)gtk_tree_view_new_with_model (
      GTK_TREE_MODEL (store));
  treeview = *treeviewptr;
  gtk_tree_view_set_rules_hint (treeview, true);
  gtk_widget_set_usize (GTK_WIDGET (treeview), 600, 300);
  gtk_container_add (GTK_CONTAINER (scrolled_window), GTK_WIDGET (treeview));

  // render the data viewer columns
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

#define COL_SETUP(Text, Index) do { \
  renderer = gtk_cell_renderer_text_new (); \
  column = gtk_tree_view_column_new_with_attributes (Text, renderer, \
      "text", Index, NULL); \
  gtk_tree_view_column_set_sort_column_id (column, Index); \
  gtk_tree_view_append_column (treeview, column); } while (0)
  
  if (listtype == NUMBERED_CHARTLIST)
    COL_SETUP (_("Label"), COL_INDEX);
  COL_SETUP   (_("Star"),  COL_NAME);
  if (listtype == SEARCHLIST)
    COL_SETUP (_("Also known as"), COL_ALTNAME);
  if (globals::chart_rules.CelestialCoords) {
    COL_SETUP (_("Right Ascension"), COL_RA);
    COL_SETUP (_("Declination"), COL_DEC);
  }
  else {
    COL_SETUP (_("Gal. Longitude"), COL_RA);
    COL_SETUP (_("Gal. Latitude"), COL_DEC);
  }
  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (store),
      COL_DEC, signed_coord_sort_func, NULL, NULL);
  COL_SETUP   ((string(_("Distance")) + " ("
		+ distance_name[globals::chart_rules.ChartUnits[1]]
		+ ")").c_str(),
	       COL_DISTANCE);
  gtk_tree_view_column_set_cell_data_func (column, renderer,
      print_double_to_text, (void *)COL_DISTANCE, 0);
  COL_SETUP   (_("Class"),     COL_SPCLASS);
  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (store),
      COL_SPCLASS, spec_class_sort_func, NULL, NULL);
  COL_SETUP   (_("Magnitude"), COL_MAGNITUDE);
  gtk_tree_view_column_set_cell_data_func (column, renderer,
      print_double_to_text, (void *)COL_MAGNITUDE, 0);
#undef COL_SETUP
  
  g_object_unref (store);
  return;
}


/* Callback to destroy info window dialog */

namespace globals {
  GtkTreeView *program_treeview = NULL;
  static GtkWidget *program_info_window = NULL;
}

void info_quit()
{
  gtk_widget_destroy(globals::program_info_window);
  globals::program_info_window = NULL;
  globals::program_treeview = NULL;
  return;
}

/* Function to update the chart information when necessary */

void update_info(GtkTreeView *treeview)
{
  int namecol = (globals::chart_rules.StarLabels == NUMBER_LABEL);
  GtkTreeViewColumn *column = gtk_tree_view_get_column (treeview, 0);
  const char *col_label = gtk_tree_view_column_get_title (column);
  
  // add or remove the "Label" column as necessary
  if ((!namecol) && strcmp(_("Label"), col_label) == 0) {
    gtk_tree_view_remove_column (treeview, column);
  }
  else if (namecol && strcmp(_("Label"), col_label)) {
    GtkCellRenderer *renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (_("Label"), renderer,
        "text", COL_INDEX, NULL);
    gtk_tree_view_column_set_sort_column_id (column, COL_INDEX);
    gtk_tree_view_insert_column (treeview, column, 0);
  }

  // re-label column headers as necessary
  column = gtk_tree_view_get_column (treeview, namecol + 1);
  gtk_tree_view_column_set_title (column,
      globals::chart_rules.CelestialCoords ?
      _("Right Ascension") : _("Gal. Longitude"));
  column = gtk_tree_view_get_column (treeview, namecol + 2);
  gtk_tree_view_column_set_title (column,
      globals::chart_rules.CelestialCoords ?
      _("Declination") : _("Gal. Latitude"));
  column = gtk_tree_view_get_column (treeview, namecol + 3);
  gtk_tree_view_column_set_title (column, (string(_("Distance")) + " ("
	+ distance_name[globals::chart_rules.ChartUnits[1]] + ")").c_str());

  // update the data
  GtkListStore *store = (GtkListStore *)gtk_tree_view_get_model (treeview);
  GtkTreeIter iter;
  gtk_list_store_clear (store);

  citerate (StarArray, *globals::chart_stararray, star_ptr) {
    StringList infolist = (*star_ptr).GetInfo(globals::chart_rules);
    starstrings::addgreek(infolist[1]);
    gtk_list_store_append (store, &iter);
    gtk_list_store_set (store, &iter,
			COL_INDEX, starmath::atoi(infolist[0]),
			COL_NAME,  infolist[1].c_str(),
			COL_RA,    infolist[2].c_str(),
			COL_DEC,   infolist[3].c_str(),
			COL_DISTANCE, (*star_ptr).GetStarDistance()
		    * distance_conversion[globals::chart_rules.ChartUnits[1]],
			COL_SPCLASS, infolist[5].c_str(),
			COL_MAGNITUDE, (*star_ptr).GetStarMagnitude(),
			-1 /* to mark end of args */);
  }

  return;
}  

/* This function displays information about the stars shown on the chart. */

void starinfo()
{
  GtkWidget *vbox, *OK;
  enum elisttype listtype = (globals::chart_rules.StarLabels == NUMBER_LABEL) ?
			    NUMBERED_CHARTLIST : CHARTLIST;

  // If the info window is already open, do nothing but put it in front
  if (globals::program_info_window) {
    gdk_window_raise(globals::program_info_window->window);
    return;
  }

  // make a window for it
  globals::program_info_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (globals::program_info_window),
			_("StarPlot: Chart Data"));
  g_signal_connect (G_OBJECT (globals::program_info_window), "destroy", 
		      G_CALLBACK (info_quit), NULL);

  // make a vbox
  vbox = gtk_vbox_new (false, 5);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
  gtk_container_add (GTK_CONTAINER (globals::program_info_window), vbox);
  gtk_widget_show (vbox);

  // set up the column list and put star information into it
  my_gtk_list_setup (vbox, &globals::program_treeview, listtype);
  update_info(globals::program_treeview);
  
  // the button to make the window go away
  my_gtk_popup_button(&OK, vbox, globals::program_info_window);
  gtk_window_set_focus (GTK_WINDOW (globals::program_info_window), OK);
  gtk_widget_show_all (globals::program_info_window);
  return;
}


/* These functions allow the user to look up star names containing a string.
 *  (admittedly, doing a `grep' of the star data files would work just as
 *  well, but it wouldn't be all nice and GUI...) */

GtkWidget *searchwindow = NULL;

struct sSearchData {
  GtkWidget   *casesensitive;
  GtkWidget   *entrybox;
  GtkTreeView *sview;
};

void search_callback(GtkWidget *widget, gpointer search_data)
{
  sSearchData   data = * (sSearchData *)search_data;
  GtkTreeView  *sview = data.sview;
  GtkListStore *store = (GtkListStore *)gtk_tree_view_get_model (sview);
  GtkTreeIter   iter;
  
  string searchstring = gtk_entry_get_text(GTK_ENTRY (data.entrybox));
  bool casesensitive = 
    (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data.casesensitive))
     == true);
  StarArray searchresults;
  StringList information;

  if (starstrings::isempty(searchstring)) return;

  gtk_list_store_clear (store);
  searchresults.Search(searchstring, globals::chart_rules.ChartFileNames,
		       globals::chart_rules, casesensitive);

  // error checking
  if (searchresults.size() >= MAX_STARS) {
    string errormsg = starstrings::ssprintf(
	_("Showing only the first %d results."), MAX_STARS);
    my_gtk_error (GTK_WINDOW (searchwindow), errormsg.c_str());
  }
  else if (!searchresults.size()) {
    my_gtk_error (GTK_WINDOW (searchwindow), starstrings::ssprintf(
	_("Sorry, no stars matching\npattern \"%s\" were found."),
	searchstring.c_str()).c_str());
    return;
  }

  citerate (StarArray, searchresults, result_ptr) {
    StringList infolist = (*result_ptr).GetInfo(globals::chart_rules);

    // where is the star name which matches the search string in the
    //  list of star characteristics?
    unsigned int nameplace = starmath::atoi(infolist[0]);
    unsigned int infolistplace = nameplace ? nameplace + 9 : 1;
    information.clear();
    information.push_back(infolist[infolistplace]);

    // if this is not the best-known star name, put that in the "aka" column;
    //  else put the second-best-known star name in that column
    information.push_back(nameplace ? infolist[1] : infolist[10]);
    starstrings::addgreek(information[0]);
    starstrings::addgreek(information[1]);

    iterate_between (StringList, infolist, j, 2, 7)
      information.push_back(*j);

    gtk_list_store_append (store, &iter);
    gtk_list_store_set (store, &iter,
			COL_NAME,     information[0].c_str(),
			COL_ALTNAME,  information[1].c_str(),
			COL_RA,	      information[2].c_str(),
			COL_DEC,      information[3].c_str(),
			COL_DISTANCE, (*result_ptr).GetStarDistance()
		    * distance_conversion[globals::chart_rules.ChartUnits[1]],
			COL_SPCLASS,  information[5].c_str(),
			COL_MAGNITUDE,(*result_ptr).GetStarMagnitude(),
			-1 /* to mark end of args */);
  }
  return;
}

/* Callback to destroy search window dialog */

void search_quit()
{
  gtk_widget_destroy(searchwindow);
  searchwindow = NULL;
  return;
}

/* create the search window dialog */

void starsearch()
{
  GtkWidget *vbox, *OK;
  GtkWidget *hbox, *casesensitive, *entrybox, *searchbutton;
  GtkTreeView *sview;
  static sSearchData search_data;
   
  // If the search window is already open, do nothing but put it in front
  if (searchwindow) {
    gdk_window_raise(searchwindow->window);
    return;
  }

  // make a window for it
  searchwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (searchwindow), _("StarPlot: Search"));
  g_signal_connect (G_OBJECT (searchwindow), "destroy", 
		      G_CALLBACK (search_quit), NULL);
  
  // make a vbox
  vbox = gtk_vbox_new (false, 5);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
  gtk_container_add (GTK_CONTAINER (searchwindow), vbox);
  gtk_widget_show (vbox);

  // set up the search input field in an hbox
  casesensitive = gtk_check_button_new_with_label (_("Case sensitive"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (casesensitive), false);
  gtk_widget_show (casesensitive);

  entrybox = gtk_entry_new();
  gtk_widget_show (entrybox);

  searchbutton = gtk_button_new_with_mnemonic (_("_Search"));
  gtk_widget_set_usize (searchbutton,
			defaults::program_button_width,
			defaults::program_button_height);
  gtk_widget_show (searchbutton);

  hbox = gtk_hbox_new (false, 5);
  gtk_box_pack_start (GTK_BOX (hbox), casesensitive, false, false, 0);
  gtk_box_pack_start (GTK_BOX (hbox), entrybox, true, true, 0); 
  gtk_box_pack_start (GTK_BOX (hbox), searchbutton, false, false, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, false, false, 0);
  gtk_widget_show (hbox);
  
  // set up the column list
  my_gtk_list_setup (vbox, &sview, SEARCHLIST);
  
  // set up the search button
  search_data.casesensitive = casesensitive;
  search_data.entrybox = entrybox;
  search_data.sview = sview;
  g_signal_connect (G_OBJECT (searchbutton), "clicked", 
		      G_CALLBACK (search_callback),
		      &search_data);
  g_signal_connect (G_OBJECT (entrybox), "activate", 
		      G_CALLBACK (search_callback),
		      &search_data);
  
  // the button to make the window go away
  my_gtk_popup_button (&OK, vbox, searchwindow);
  gtk_window_set_focus (GTK_WINDOW (searchwindow), OK);
  gtk_widget_show_all (searchwindow);
  return;
}

// ---------------------------------------------------------------------------
/* This function displays a dialog allowing to calculate the distance
 * between two stars (or arbitrary coordinates): */

/* Callback function that actually calculates the distance: */
static void calculate_star_distance(GtkWidget *widget, gpointer star_data)
{
  GtkWidget ** coords = (GtkWidget **)star_data;

  int mult[2] = { 1, -1 };
  Vector3 result;

  for (unsigned i = 0; i < 2; i++) {
    Vector3 v = SolidAngle( starstrings::strs_to_ra(
			      gtk_entry_get_text (GTK_ENTRY (coords[8*i + 0])),
			      gtk_entry_get_text (GTK_ENTRY (coords[8*i + 1])),
			      gtk_entry_get_text (GTK_ENTRY (coords[8*i + 2])),
			      globals::chart_rules.CelestialCoords),
			    starstrings::strs_to_dec(
			      gtk_entry_get_text (GTK_ENTRY (coords[8*i + 3])),
			      gtk_entry_get_text (GTK_ENTRY (coords[8*i + 4])),
			      gtk_entry_get_text (GTK_ENTRY (coords[8*i + 5]))
			    )
			  ).toCartesian(starmath::atof(
			      gtk_entry_get_text(GTK_ENTRY (coords[8*i + 6])))
		    / distance_conversion[globals::chart_rules.ChartUnits[1]]);
    result += mult[i] * v;
  }

  string dist = starstrings::distance_to_str(result.magnitude(),
					     globals::chart_rules.ChartUnits);
  gtk_entry_set_text (GTK_ENTRY (coords[2*8]), dist.c_str());
}

void stardistance(void)
{
  GtkWidget *window, *main_hbox, *main_vbox, *title;
  GtkWidget *star_frame[2], *star_vbox[2];
  GtkWidget *OK, *calculate, *results_hbox, *results_label;
  static GtkWidget * star_coords[2*8 + 1];

  // The window
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_resizable (GTK_WINDOW (window), false);
  gtk_window_set_title (GTK_WINDOW (window), _("StarPlot: Calculate Distance"));
  gtk_window_set_modal (GTK_WINDOW (window), true);
  g_signal_connect (G_OBJECT (window), "destroy",
                      G_CALLBACK (gtk_widget_destroy), NULL);

  // Pack it vertically
  main_vbox = gtk_vbox_new (false, 5);
  gtk_container_set_border_width (GTK_CONTAINER (main_vbox), 10);
  gtk_container_add (GTK_CONTAINER (window), main_vbox);
  gtk_widget_show (main_vbox);

  // Add explanatory title
  title = gtk_label_new (_(
"Define the two positions between which to calculate the distance.\n\
(The default for the first position is the center of the current chart.)"));
  gtk_box_pack_start (GTK_BOX (main_vbox), title, false, false, 0);
  gtk_misc_set_alignment (GTK_MISC (title), 0.5F, 0.0F);
  gtk_widget_show (title);

  // Set up star frames horizontally
  main_hbox = gtk_hbox_new (true, 20);
  gtk_container_set_border_width (GTK_CONTAINER (main_hbox), 10);
  gtk_box_pack_start (GTK_BOX (main_vbox), main_hbox, false, false, 0);
  gtk_widget_show (main_hbox);

  // Set up star frames
  for (unsigned i = 0; i < 2; i++) {
    string s = starstrings::ssprintf(_("Coordinates of position %d:"), i+1);
    star_frame[i] = gtk_frame_new (s.c_str());
    gtk_box_pack_start (GTK_BOX (main_hbox), star_frame[i], false, false, 0);
    gtk_frame_set_label_align (GTK_FRAME (star_frame[i]), 0.5F, 0.0F);
    gtk_frame_set_shadow_type (GTK_FRAME (star_frame[i]), GTK_SHADOW_NONE);
    gtk_widget_show (star_frame[i]);

    star_vbox[i] = gtk_vbox_new (false, 5);
    gtk_container_add (GTK_CONTAINER (star_frame[i]), star_vbox[i]);
    gtk_container_set_border_width (GTK_CONTAINER (star_vbox[i]), 5);
    gtk_widget_show (star_vbox[i]);

    my_gtk_star_coordinates(&star_coords[8*i], star_vbox[i]);
  }

  // Set the default for the second position to be the origin.
  for (unsigned i = 0; i < 6; i++)
    gtk_entry_set_text (GTK_ENTRY (star_coords[8 + i]), "00");
  gtk_entry_set_text (GTK_ENTRY (star_coords[8+6]), "0");

  gtk_entry_set_text (GTK_ENTRY (star_coords[0+7]), 
      (string("[") + _("Chart Center") + string("]")).c_str());
  gtk_entry_set_text (GTK_ENTRY (star_coords[8+7]), "Sun");

  // Add calculate button and results field
  results_hbox = gtk_hbox_new (false, 20);
  gtk_container_set_border_width (GTK_CONTAINER (results_hbox), 10);
  gtk_box_pack_start (GTK_BOX (main_vbox), results_hbox, false, false, 0);
  gtk_widget_show (results_hbox);

  GtkWidget * spacer = gtk_label_new ("");
  gtk_box_pack_start (GTK_BOX (results_hbox), spacer, true, false, 0);
  gtk_widget_show (spacer);

  calculate = gtk_button_new_with_mnemonic (_("_Calculate distance"));
  gtk_box_pack_start (GTK_BOX (results_hbox), calculate, false, false, 10);
  gtk_widget_show (calculate);

  results_label = gtk_label_new (_("Distance between positions is:"));
  gtk_box_pack_start (GTK_BOX (results_hbox), results_label, false, false, 0);
  gtk_misc_set_alignment (GTK_MISC (results_label), 1.0F, 0.5F);
  gtk_widget_show (results_label);

  star_coords[2*8] = gtk_entry_new();
  gtk_box_pack_start (GTK_BOX (results_hbox), star_coords[2*8], false,false, 0);
  gtk_entry_set_editable (GTK_ENTRY (star_coords[2*8]), false);
  gtk_widget_show (star_coords[2*8]);

  spacer = gtk_label_new ("");
  gtk_box_pack_start (GTK_BOX (results_hbox), spacer, true, false, 0);
  gtk_widget_show (spacer);
  
  g_signal_connect (G_OBJECT (calculate), "clicked",
                      G_CALLBACK (calculate_star_distance), star_coords);

  my_gtk_popup_button(&OK, main_vbox, window);
  
  // Finally, set the window policy and show it
  gtk_window_set_focus (GTK_WINDOW (window), OK);
  gtk_widget_show (window);

  return;
}
